@ECHO OFF
:: ============================================================================
:: This script serves as a guide defining best-practice configurables for a
:: production environment.  See documentation regarding configurables here:
:: https://help.perforce.com/helix-core/server-apps/cmdref/current/Content/CmdRef/configurables.alphabetical.html
:: 
:: Copyright and license info is available in the LICENSE file included with
:: the Server Deployment Package (SDP), and also available online:
:: https://swarm.workshop.perforce.com/projects/perforce-software-sdp/view/main/LICENSE
:: ----------------------------------------------------------------------------
REM Set P4PORT and P4USER and run p4 login before running this script.

:: Verify instance value
if %1x==x (
    echo Error: The SDP instance argument is required.
    echo .
    exit /b 1
)

SET instance=%1
call C:\p4\common\bin\p4env.bat %1

@ECHO ON

C:\p4\common\bin\svcinst.exe start -n p4_%instance%

:: The server.depot.root configurable is valid for P4D 2014.1+
p4 configure set server.depot.root=c:\p4\%instance%\depots
:: The server.rolechecks configurable is valid for P4D 2011.1+
p4 configure set server.rolechecks=1
p4 configure set journalPrefix=%CHECKPOINTS_DIR%\p4_%instance%
:: The db.peeking configurable is valid for P4D 2013.3+
p4 configure set db.peeking=3
p4 configure set dm.user.noautocreate=2
p4 configure set dm.info.hide=1
p4 configure set dm.user.setinitialpasswd=0
p4 configure set dm.user.resetpassword=1

p4 configure set filesys.P4ROOT.min=5G
p4 configure set filesys.P4JOURNAL.min=5G
p4 configure set filesys.P4LOG.min=5G
p4 configure set filesys.depot.min=5G
p4 configure set filesys.TEMP.min=5G

p4 configure set server=4
p4 configure set monitor=2

:: For P4D 2015.2+, make client views global in a commit/edge environment.
p4 configure set server.global.client.views=1

:: For P4D 2013.2+, setting db.reorg.disable=1, which turns off
:: dynamic database reorg, has been shown to significantly improve
:: performance when Perforce databases (db.* files) are stored on
:: some solid state storage devices, while not making a difference
:: on others.
p4 configure set db.reorg.disable=1

:: Performance Tracking as required by P4Promtheus.
p4 configure set track=1

:: For P4D 2017.2.1594901 or greater, enable net.autotune.  For net.autotune
:: to take effect, it must be enabled on both sides of a connection.  So, to
:: get the full benefit, net.autotune must be enabled on all brokers, proxies,
:: and clients.  See this KB article for details on fully enabling net.autotune:
:: https://portal.perforce.com/s/article/15368
::
:: For connections in which net.autotune is not enabled, the p4d default value
:: of net.tcpsize takes effect.
::
:: When P4D is older than 2014.2 but less than 2017.2.1594901, set net.tcpsize
:: to 512k.  In 2014.2, the default value for net.tcpsize became 512k, a
:: reasonable default, so it should not be set explicitly.  Also, there are
:: indications it can reduce performance if set when not needed.
p4 configure unset net.autotune
p4 configure unset net.tcpsize

:: For connections in which net.autotune is not enabled, the p4d default value
:: of net.tcpsize takes effect.
::
:: When P4D is older than 2014.2 but less than 2017.2.1594901, set net.tcpsize
:: to 512k.  In 2014.2, the default value for net.tcpsize became 512k, a
:: reasonable default, so it should not be set explicitly.  Also, there are
:: indications it can reduce performance if set when not needed.
p4 configure unset net.tcpsize

:: For P4D 2016.2.1468155+, set db.monitor.shared = max value.
:: This is the number of 8k pages to set aside for monitoring,
:: which requires pre-allocation of sufficient RAM.  The default
:: is 256, or 2MB, enough for about 128 active/concurrent processes.
:: The max as of 2016.2 is 4096.  Setting db.monitor.shared=0
:: causes the db.monitor on disk to be used instead, which can
:: potentially be a bottleneck.
p4 configure set db.monitor.shared=4096

p4 configure set net.backlog=2048

p4 configure set lbr.bufsize=1M
p4 configure set lbr.autocompress=1
p4 configure set filesys.bufsize=1M

p4 configure set serverlog.file.1=%LOGS%\auth.csv
p4 configure set serverlog.retain.1=%KEEPLOGS%

p4 configure set serverlog.file.3=%LOGS%\errors.csv
p4 configure set serverlog.retain.3=%KEEPLOGS%

:: The following are useful if using threat detection based on P4AUDIT
:: logs or if those logs are otherwise desired. These are not enabled
:: by default as they have special considerations for performance,
:: storage, retention, and possibly external processing.
:: p4 configure set serverlog.file.4=%LOGS_DIR%\audit.csv

p4 configure set serverlog.file.7=%LOGS_DIR%\events.csv
p4 configure set serverlog.retain.7=%KEEPLOGS%

p4 configure set serverlog.file.8=%LOGS_DIR%\integrity.csv
p4 configure set serverlog.retain.8=%KEEPLOGS%

:: Add a custom trigger for tracking trigger events:
p4 configure set serverlog.file.11=%LOGS_DIR%\triggers.csv
p4 configure set serverlog.retain.11=%KEEPLOGS%

:: Net Keepalives
p4 configure set net.keepalive.count=9
p4 configure set net.keepalive.disable=0
p4 configure set net.keepalive.idle=180
p4 configure set net.keepalive.interval=15

@echo "Creating a depot named 'spec' of type 'spec'.
p4 depot -i < spec.depot.p4s

@echo "Creating a depot named 'unload' of unload 'unload'.
p4 depot -i < unload.depot.p4s

:: For p4d 2013.2+ -Turn off max* commandline overrides.
p4 configure set server.commandlimits=2

@echo See: https://portal.perforce.com/s/article/3867
p4 configure set rpl.checksum.auto=1
p4 configure set rpl.checksum.change=2
p4 configure set rpl.checksum.table=1

:: Define number of login attempts before there is a delay, to thwart
:: automated password crackers.  Default is 3; set to a higher value to
:: be more friendly to humans without compromising the protection.
p4 configure set dm.user.loginattempts=7

:: For p4d 2016.1 Patch 5+
:: Enable a server with an expired temp license to start, albeit with limited
:: functionality, so that license expiry doesn't make it impossible to perform
:: license management via the front-door.  This configurable allows the server
:: to be started regardless of a bad license, though users will still be blocked
:: by license invalid messages.  Perpetual commercial licenses never expire;
:: this configurable will not affect those.
p4 configure set server.start.unlicensed=1

:: Starting with p4d 2015.1 Patch 5, disallow P4EXP v2014.2 (a client
:: version known to misbehave) from connecting to the server.
:: See:  http://portal.perforce.com/articles/KB/15014
p4 configure set rejectList="P4EXP,version=2014.2"

:: For p4d 2011.1 thru 2015.1, set rpl.compress=3.  For p4d 2015.2+, set
:: rpl.compress=4.  This setting compresses journal data only, which is
:: almost always advantageous as it compresses well, while avoiding
:: compression of archive data, which is a mixed bag in terms of performance
:: benefits, and potentially a net negative.
p4 configure set rpl.compress=4

:: Starting with p4d 2016.2, enable these features.
p4 configure set server.locks.global=1
p4 configure set proxy.monitor.level=3

:: Enable faster resubmit after failed submit.
p4 configure set submit.noretransfer=1

:: Recommended for Swarm
p4 configure set dm.shelve.promote=1
p4 configure set dm.keys.hide=2
p4 configure set filetype.bypasslock=1

:: Starting with p4d 2018.2 (as tech-preview, 2019.2 for GA), add best
:: practices for Extensions.
p4 configure set server.extensions.dir=c:\p4\%instance%\logs\p4-extensions

:: Set configurables to optimize for Helix Authentication Service (HAS)
:: deployment. These will also affect behavior of older `auth-check-sso`
:: triggers.
:: For p4d 2018.2+
p4 configure set auth.sso.allow.passwd=1
p4 configure set auth.sso.nonldap=1

:: Enable parallelization.
p4 configure set net.parallel.max=10
p4 configure set net.parallel.threads=4

:: Limit max parallel syncs.
p4 configure set net.parallel.sync.svrthreads=150

:: Enable partitioned clients.
p4 configure set client.readonly.dir=client.readonly.dir
p4 configure set client.sendq.dir=client.readonly.dir

:: Starting with p4d 2016.1, use auth.id to simplify ticket handling.
:: After setting auth.id, login again.
p4 configure set rpl.forward.login=1
p4 configure set auth.id=p4_%instance%
p4login.bat %1

:: Set SDP version identifying info.
p4 counter SDP "%DATE%"

:: The filesys.atomic.rename feature is enabled by default in P4D 2023.2.
:: This feature is preferred when it can be used, but does not work in
:: all storage configurations.  It is set to zero here to ensure journal
:: rotations and checkpoints always work.  If you determine that journal
:: rotations work with atomic rename enabled in the current environment,
:: this configurable can be set to 1.
p4 configure set filesys.atomic.rename=0

:: Basic security features.
p4 configure set run.users.authorize=1
p4 configure set dm.user.hideinvalid=1
p4 configure set security=4

:: Enable real time monitoring with 'p4 monitor rt'.
p4 configure set rt.monitorfile=monfile.mem

:: Restart to ensure all configurable changes take effect.
C:\p4\common\bin\svcinst.exe stop -n p4_%instance%
C:\p4\common\bin\svcinst.exe start -n p4_%instance%
p4login.bat %1

@echo The net.autotune value has been set on the server.  To get the full benefit, it must also be
@echo enabled on proxies, brokers, and clients as well.
