#!/bin/bash
set -u

declare ThisScript="${0##*/}"
declare Version=1.0.1
declare -i ErrorCount=0
declare -i Debug=${SDP_DEBUG:-0}
declare EdgeHost=syd-helix-04
declare StandbyHost=bos-helix-02
declare -i EdgeLogCount=0
declare -a EdgeLogList
declare -i StandbyLogCount=0
declare -a StandbyLogList

function msg () { echo -e "$*"; }
function dbg () { [[ "$Debug" -eq 0 ]] || msg "DEBUG: $*"; }
function errmsg () { msg "\\nError: ${1:-Unknown Error}\\n"; ErrorCount+=1; }
function bail () { errmsg "${1:-Unknown Error}"; exit "$ErrorCount"; }

msg "Start test [$ThisScript v$Version]: Do live checkpoint on edge and standby."

# Setup: Create site config dir if needed
mkdir -p /p4/common/site/config
ssh "$EdgeHost" mkdir -p /p4/common/site/config
ssh "$StandbyHost" mkdir -p /p4/common/site/config

msg "Configure for serial checkpoint ops on commit, edge, and standby."

{
   echo export DO_PARALLEL_CHECKPOINTS=0
   echo KEEPCKPS=2
   echo KEEPJNLS=2
   echo KEEPCKPS=2
} > /p4/common/site/config/p4_1.vars.local

rsync /p4/common/site/config/p4_1.vars.local "$EdgeHost":/p4/common/site/config/p4_1.vars.local ||\
   errmsg "Failed to copy /p4/common/site/config/p4_1.vars.local."
rsync /p4/common/site/config/p4_1.vars.local "$StandbyHost":/p4/common/site/config/p4_1.vars.local ||\
   errmsg "Failed to copy /p4/common/site/config/p4_1.vars.local."

#------------------------------------------------------------------------------
msg "Doing live checkpoint on edge."
msg "Running: live_checkpoint.sh 1 on edge host $EdgeHost"

msg "\\nRunning: ssh $EdgeHost /p4/common/bin/live_checkpoint.sh 1"
ssh "$EdgeHost" /p4/common/bin/live_checkpoint.sh 1 ||\
   errmsg "Remote call to live_checkpoint.sh on edge host $EdgeHost indicated an error."

EdgeLogList[EdgeLogCount]="/p4/1/logs/checkpoint.log"
EdgeLogCount+=1

#------------------------------------------------------------------------------
msg "Doing live checkpoint on standby."
msg "Running: live_checkpoint.sh 1 on standby host $StandbyHost"

msg "\\nRunning: ssh $StandbyHost /p4/common/bin/live_checkpoint.sh 1"
ssh "$StandbyHost" /p4/common/bin/live_checkpoint.sh 1 ||\
   errmsg "Remote call to live_checkpoint.sh on standby host $StandbyHost indicated an error."

StandbyLogList[StandbyLogCount]="/p4/1/logs/checkpoint.log"
StandbyLogCount+=1

#------------------------------------------------------------------------------
msg "Checking for syntax and unbound errors in scripts used in this test on edge host."
for log in "${EdgeLogList[@]}"; do
   msg "Checking for 'unbound variable' in: $EdgeHost:$log"
   ssh -q "$EdgeHost" "grep ':unbound variable' $log" && \
      errmsg "Found 'unbound variable' error in $log"
   msg "Checking for 'syntax error' in: $EdgeHost:$log"
   ssh -q "$EdgeHost" "grep ':syntax error' $log" && \
      errmsg "Found 'syntax error' in $log"
done

msg "Checking for syntax and unbound errors in scripts used in this test on standby host."
for log in "${StandbyLogList[@]}"; do
   msg "Checking for 'unbound variable' in: $StandbyHost:$log"
   ssh -q "$StandbyHost" "grep ':unbound variable' $log" && \
      errmsg "Found 'unbound variable' error in $log"
   msg "Checking for 'syntax error' in: $StandbyHost:$log"
   ssh -q "$StandbyHost" "grep ':syntax error' $log" && \
      errmsg "Found 'syntax error' in $log"
done

exit "$ErrorCount"
