#!/bin/bash
set -u

declare ThisScript="${0##*/}"
declare Version=1.0.2
declare -i ErrorCount=0
declare -i Debug=${SDP_DEBUG:-0}
declare -i LocalLogCount=0
declare -a LocalLogList
declare -i RemoteLogCount=0
declare -a RemoteLogList
declare -a RemoteLogHost
declare StandbyHost=bos-helix-02
declare EdgeHost=syd-helix-04
declare RemoteHost=
declare RemoteLog=

function msg () { echo -e "$*"; }
function dbg () { [[ "$Debug" -eq 0 ]] || msg "DEBUG: $*"; }
function errmsg () { msg "\\nError: ${1:-Unknown Error}\\n"; ErrorCount+=1; }
function bail () { errmsg "${1:-Unknown Error}"; exit "$ErrorCount"; }

msg "Start test [$ThisScript v$Version]: Load latest serial checkpoint."

# Setup: Create site config dir if needed
mkdir -p /p4/common/site/config
ssh "$EdgeHost" mkdir -p /p4/common/site/config

msg "\\nConfigure for serial checkpoint ops on commit and edge."

{
   echo export DO_PARALLEL_CHECKPOINTS=0
   echo KEEPCKPS=2
   echo KEEPJNLS=2
   echo KEEPCKPS=2
} > /p4/common/site/config/p4_1.vars.local

msg "Pushing checkpoint config to edge."
rsync /p4/common/site/config/p4_1.vars.local "$EdgeHost:/p4/common/site/config/p4_1.vars.local" ||\
   errmsg "Failed to copy /p4/common/site/config/p4_1.vars.local to $EdgeHost."

msg "Do a live checkpoint on the commit."
live_checkpoint.sh 1
LocalLogList[LocalLogCount]="/p4/1/logs/checkpoint.log"
LocalLogCount+=1

msg "Pushing /p4/1/checkpoints to standby."
rsync -a /p4/1/checkpoints/ "$StandbyHost":/p4/1/checkpoints ||\
   errmsg "Test setup error pushing checkpoints to standby $StandbyHost."

msg "\\nLoad latest serial checkpoint on standby."

msg "Running: ssh $StandbyHost load_checkpoint.sh -latest -R -b -y -L /p4/1/logs/load.log"
ssh "$StandbyHost" "load_checkpoint.sh -latest -R -b -y -L /p4/1/logs/load.log"
RemoteLogHost[RemoteLogCount]="$StandbyHost"
RemoteLogList[RemoteLogCount]="/p4/1/logs/load.log"
RemoteLogCount+=1

msg "\\nTake serial checkpoint on edge."

msg "Running: ssh $EdgeHost request_replica_checkpoint.sh 1 -now"
ssh "$EdgeHost" "request_replica_checkpoint.sh 1 -now"
RemoteLogHost[RemoteLogCount]="$EdgeHost"
RemoteLogList[RemoteLogCount]="/p4/1/logs/request_checkpoint.log"
RemoteLogCount+=1

# Give the checkpoint time to complete. 3 secs should be enough ...
### TO DO: Come up with a more reliable way to give it time to complete
### (but also not hang if it doesn't complete for some reason).
sleep 3

msg "Running: ssh $EdgeHost load_checkpoint.sh -latest -R -b -y -L /p4/1/logs/load.log"
ssh "$EdgeHost" "load_checkpoint.sh -latest -R -b -y -L /p4/1/logs/load.log"
RemoteLogHost[RemoteLogCount]="$EdgeHost"
RemoteLogList[RemoteLogCount]="/p4/1/logs/load.log"
RemotLogCount+=1

msg "Checking for syntax and unbound errors in scripts used in this test."
for log in "${LocalLogList[@]}"; do
   msg "Checking for 'unbound variable' in: $log"
   grep ':unbound variable' "$log" && \
      errmsg "Found 'unbound variable' error in $log"
   msg "Checking for 'syntax error' in: $log"
   grep ':syntax error' "$log" && \
      errmsg "Found 'syntax error' in $log"
done

msg "Checking for syntax and unbound errors in scripts used in this test on remote hosts."
for ((i=0; i < RemoteLogCount; i++)); do
   RemoteHost="${RemoteLogHost[i]}"
   RemoteLog="${RemoteLogList[i]}"
   msg "Checking for 'unbound variable' in: $RemoteHost:$RemoteLog"
   ssh -q "$RemoteHost" "grep ':unbound variable' $RemoteLog" && \
      errmsg "Found 'unbound variable' error in $RemoteLog"
   msg "Checking for 'syntax error' in: $RemoteHost:$RemoteLog"
   ssh -q "$RemoteHost" "grep ':syntax error' $RemoteLog" && \
      errmsg "Found 'syntax error' in $RemoteLog"
done

exit "$ErrorCount"
