#!/bin/bash
set -u

declare ThisScript="${0##*/}"
declare Version=1.0.2
declare -i ErrorCount=0
declare -i Debug=${SDP_DEBUG:-0}
declare LinkTarget=
declare TmpFile=
declare OfflineDB=

function msg () { echo -e "$*"; }
# shellcheck disable=SC2317
function dbg () { [[ "$Debug" -eq 0 ]] || msg "DEBUG: $*"; }
function errmsg () { msg "\\nError: ${1:-Unknown Error}\\n"; ErrorCount+=1; }
function bail () { errmsg "${1:-Unknown Error}"; exit "$ErrorCount"; }

msg "Start test [$ThisScript v$Version]: Load checkpoint man page."

TmpFile=$(mktemp)

# shellcheck disable=SC1091
source /p4/common/bin/p4_vars 1 ||\
   bail "Could not do: source /p4/common/bin/p4_vars 1"

OfflineDB="$P4HOME/offline_db"

if p4 depot --exists -o Perforce > /dev/null 2>&1; then
   msg "Remote depot 'Perforce' exists. Removing it with: p4 depot -d Perforce"
   p4 depot -d Perforce ||\
      bail "Failed to delete depot 'Perforce'.  Aborting this test due to unexpected pre-start setup failure."
else
   msg "Verified: Remote depot 'Perforce' does not exist."
fi

msg "Test setup: Removing partitioned clients data from offline_db. Running:\\nrm -rf \"$OfflineDB/client.readonly.dir\""
rm -rf "$OfflineDB/client.readonly.dir" ||\
   bail "Failed to remove $OfflineDB/client.readonly.dir.  Aborting this test due to unexpected pre-start setup failure."

LinkTarget=$(readlink "${P4ROOT}")
if [[ "$LinkTarget" == *db2 ]]; then
   msg "Verified: Pre-swap P4ROOT symlink points to db2."
else
   bail "Pre-swap P4ROOT does NOT point to db2. Aborting this test due to unexpected pre-start conditions."
fi

LinkTarget=$(readlink "${P4ROOT%/root}/offline_db")
if [[ "$LinkTarget" == *db1 ]]; then
   msg "Verified: Pre-swap offline_db symlink points to db1."
else
   bail "Pre-swap offline_db symlink does NOT point to db1. Aborting this test due to unexpected pre-start conditions."
fi

msg "Creating partitioned client."
p4 --field Type=partitioned --field Root=/p4/1/tmp/ws/MyPartitionedClient client -o MyPartitionedClient | p4 client -i

msg "Flusing partitioned client, running: p4 -s -c MyPartitionedClient flush -q"
p4 -s -c MyPartitionedClient flush -q || errmsg "Flush failed. Aborting this test due to unexpected pre-start conditions."

msg "Running: refresh_P4ROOT_from_offline_db.sh"
refresh_P4ROOT_from_offline_db.sh ||\
   errmsg "refresh_P4ROOT_from_offline_db.sh reported non-zero exit code in test expecting 0 exit code."

LinkTarget=$(readlink "${P4ROOT}")
if [[ "$LinkTarget" == *db1 ]]; then
   msg "Verified: Post-swap P4ROOT symlink points to db1."
else
   errmsg "Post-swap P4ROOT does NOT point to db1."
fi

LinkTarget=$(readlink "${P4ROOT%/root}/offline_db")
if [[ "$LinkTarget" == *db2 ]]; then
   msg "Verified: Post-swap offline_db symlink points to db2."
else
   bail "Post-swap offline_db symlink does NOT point to db2."
fi

sleep 1

msg "Ensuring partitioned client data survied the swap intact.\\nRunning: p4 -s -c MyPartitionedClient flush"
p4 -s -c MyPartitionedClient flush > "$TmpFile" 2>&1
cat "$TmpFile"

if grep -q 'up-to-date' "$TmpFile"; then
   msg "Verified: Partitioned client 'have' data survived the db swap intact."
else
   errmsg "Partitioned client 'have' data did not appear to survive the db swap intact."
fi

exit "$ErrorCount"
