#!/bin/bash
 
# Progress monitoring script for batch_verify process
# Shows current progress against total files to process
 
# Find the latest batch_verify log file
LATEST_LOG=$(ls -t batch_verify_*.log 2>/dev/null | head -n 1)
 
if [[ -z "$LATEST_LOG" ]]; then
    echo "Error: No batch_verify_*.log files found"
    exit 1
fi
 
# Default input file (can be overridden with -f parameter)
INPUT_FILE="files_to_fstat.txt"
 
# Parse command line arguments
while getopts "f:h" opt; do
    case $opt in
        f) INPUT_FILE="$OPTARG" ;;
        h)
            echo "Usage: $0 [-f input_file]"
            echo ""
            echo "Options:"
            echo "  -f <file>  Input file to check total count against (default: files_to_fstat.txt)"
            echo ""
            echo "Shows progress of batch verification by comparing processed files"
            echo "from the latest batch_verify_*.log against total files in input file."
            exit 0
            ;;
        *)
            echo "Usage: $0 [-f input_file]"
            exit 1
            ;;
    esac
done
 
# Check if input file exists
if [[ ! -f "$INPUT_FILE" ]]; then
    echo "Error: Input file '$INPUT_FILE' not found"
    exit 1
fi
 
# Get total files from input file
TOTAL_FILES=$(wc -l < "$INPUT_FILE")
 
# Extract processed files count from log using awk
PROCESSED_FILES=$(cat "$LATEST_LOG" | awk '/completed successfully/{s += substr($7, 2)} END {print s+0}')
 
# Handle case where no completed batches yet
if [[ -z "$PROCESSED_FILES" ]] || [[ "$PROCESSED_FILES" == "0" ]]; then
    PROCESSED_FILES=0
fi
 
# Calculate percentage
if [[ $TOTAL_FILES -gt 0 ]]; then
    PERCENTAGE=$(awk "BEGIN {printf \"%.2f\", ($PROCESSED_FILES / $TOTAL_FILES) * 100}")
else
    PERCENTAGE="0.00"
fi
 
# Display results
echo "=== Batch Verification Progress ==="
echo "Date/Time: $(date '+%Y-%m-%d_%H:%M:%S')"
echo "Log file: $LATEST_LOG"
echo "Input file: $INPUT_FILE"
echo ""
echo "Processed: $PROCESSED_FILES files"
echo "Total:     $TOTAL_FILES files"
echo "Progress:  $PERCENTAGE%"
echo ""
 
# Show remaining files
REMAINING=$((TOTAL_FILES - PROCESSED_FILES))
echo "Remaining: $REMAINING files"
 
# Show recent activity from log (last 5 lines)
echo ""
echo "=== Recent Activity ==="
tail -n 5 "$LATEST_LOG"
