#!/bin/bash
#------------------------------------------------------------------------------
# Repro for:  jobNNNNN
set -u

# Usage:
# ./repro.sh [-f] 2>&1
#
# Scenario: Delete of Deleted File

export P4CONFIG=.p4config.repro
export P4ENVIRO=/dev/null/.p4enviro
declare ThisScript=${0##*/}
declare Version=1.1.0
declare -i ErrorCount=0
declare AppHome="$PWD"
declare H1="=============================================================================="
declare Log="${ThisScript%.sh}.log"
declare CmdLog=

# Micro functions.
function msg () { echo -e "${1:-Hi}"; }
function errmsg () { msg "\\nError: ${1:-Unknown Error}"; ErrorCount+=1; }
function bail () { errmsg "${1:-Unknown Error}"; exit "$ErrorCount"; }
function cmd () { msg "${2:-Executing command: $1}"; $1; return $?; }

declare -i shiftArgs=0
declare -i Force=0
declare -i Scenario=1
declare ScenarioTitle="Delete Of Deleted File"

#------------------------------------------------------------------------------
# Command Line Args
set +u; while [[ $# -gt 0 ]]; do
   case $1 in
      (-f) Force=1;;
      (-*) bail "Usage error: Unknown option ($1).";;
      (*) bail "Usage error: Unknown parameter ($1).";;
   esac

   # Shift (modify $#) the appropriate number of times.
   shift; while [[ $shiftArgs -gt 0 ]]; do
      [[ $# -eq 0 ]] && bail "Bad usage."
      shiftArgs=$shiftArgs-1
      shift
   done
done
set -u

ReproDir=/tmp/repro

if [[ "$Log" != off ]]; then
   touch "$Log" || bail "Couldn't touch log file [$Log]."

   # Redirect stdout and stderr to a log file.
   exec > >(tee "$Log")
   exec 2>&1
fi

CmdLog=$(mktemp)

msg "Started ${0##*/} v$Version at $(date) in $AppHome."

msg "ReproDir=$ReproDir"
[[ -d "$ReproDir" && "$Force" -eq 1 ]] && /bin/rm -rf "$ReproDir"
[[ -d "$ReproDir" ]] && bail "Old repro dir [$ReproDir] exists."

mkdir "$ReproDir" || bail "Could not do: mkdir \"$ReproDir\""
cd "$ReproDir" || bail "Could not do: cd \"$ReproDir\""

msg "==============================================================================\nScenario $Scenario: $ScenarioTitle\n"
msg "\nPreliminary info: Show versions of p4/p4d on the PATH:"
cmd "p4 -V"
cmd "p4d -V"

msg "\nPreliminary setup: Spin up a local repo."
cd "$ReproDir" || bail "Could not do: cd \"$ReproDir\""
msg "Operating in: $PWD"
cmd "p4 init -C0 -n"

msg "Client spec before tweak:"
p4 client -o | grep -v ^#
msg "\\nChanging client 'SubmitOptions:' field value to 'leaveunchanged'."
p4 --field SubmitOptions=leaveunchanged client -o | p4 client -i
msg "Client spec after tweak:"
p4 client -o | grep -v ^#

msg "Starting file operations:"
cmd "touch foo"
cmd "p4 add foo"
cmd "p4 submit -d Add."

msg "Generating client ws2."
p4 client -S //stream/main -o ws2 | p4 client -i
p4 client -S //stream/main -o ws3 | p4 client -i
cmd "rm -f foo"
cmd "p4 -c ws2 sync"
cmd "p4 -c ws3 sync"
cmd "p4 delete foo"
cmd "p4 submit -d DeleteMeOnce"

# For just this one command, use a separte log so that our
# 'grep' below isn't subject to a race condition with our
# 'tee' log mechanism and output/log buffering.
msg "Executing command: p4 -c ws2 delete foo"
p4 -c ws2 delete foo > "$CmdLog" 2>&1
p4 -c ws2 submit -d DeleteMeTwice
cat "$CmdLog"

p4 -c ws3 delete foo > "$CmdLog" 2>&1
p4 -c ws3 submit -d DeleteMeThreeTimes
cat "$CmdLog"

cmd "p4 filelog //stream/main/foo"

msg "${H1}\\nThe Result:"
if [[ "$ErrorCount" -eq 1 ]]; then
   msg "Yay! We are getting the preferred result, i.e. that the attempt to delete an already-deleted-and-then-submitted file results in an error."
elif [[ "$ErrorCount" -eq 0 ]]; then
   if grep -q 'warning: delete of deleted file' "$CmdLog"; then
      rm -f "$CmdLog"
      msg "\\nThis is the currently expected p4d behvaior, allowing the delete of a deleted file with just a warning, 'warning: delete of deleted file' .  Is there a valid use case for allowing a delete of a deleted file just because workspaces were not in sync? Is it risky to change this behaviour?\\n\\nNote that a workspace SubmitOptions value of leaveunchange and revertunchanged have no impact; they do not prevent the double-delete - or triple ..."
   else
      rm -f "$CmdLog"
      bail "An unexpected situation has arisen. There were no errors, but we did not get the warning about the deletion of a deleted file."
   fi
else
   bail "An unexpected number of errors ($ErrorCount) were encountered; exactly 1 was expected."
fi

exit "$ErrorCount"
