#!/bin/bash
#------------------------------------------------------------------------------
# Repro for:  jobNNNNN
set -u

# Usage:
# ./repro.sh [-f] 2>&1 | tee repro.log
#
# Scenario: Fetch Partial Rename

export P4CONFIG=.p4config
export P4ENVIRO=/dev/null/.p4enviro
declare Version=1.0.0
declare -i ErrorCount=0
declare AppHome="$PWD"
declare H1="=============================================================================="

# Micro functions.
function msg () { echo -e "${1:-Hi}"; }
function errmsg () { msg "\\nError: ${1:-Unknown Error}"; ErrorCount+=1; }
function bail () { errmsg "${1:-Unknown Error}"; exit "$ErrorCount"; }
function cmd () { msg "${2:-Executing command: $1}"; $1; return $?; }

declare -i shiftArgs=0
declare -i Force=0
declare -i Scenario=1
declare ScenarioTitle="Fetch Partial Rename"

#------------------------------------------------------------------------------
# Command Line Args
set +u; while [[ $# -gt 0 ]]; do
   case $1 in
      (-f) Force=1;;
      (-*) bail "Usage error: Unknown option ($1).";;
      (*) bail "Usage error: Unknown parameter ($1).";;
   esac

   # Shift (modify $#) the appropriate number of times.
   shift; while [[ $shiftArgs -gt 0 ]]; do
      [[ $# -eq 0 ]] && bail "Bad usage."
      shiftArgs=$shiftArgs-1
      shift
   done
done
set -u

ReproDir=/tmp/repro

msg "Started ${0##*/} v$Version at $(date)."

msg "ReproDir=$ReproDir"
[[ -d "$ReproDir" && "$Force" -eq 1 ]] && /bin/rm -rf "$ReproDir"
[[ -d "$ReproDir" ]] && bail "Old repro dir [$ReproDir] exists."

mkdir "$ReproDir" || bail "Could not do: mkdir \"$ReproDir\""
cd "$ReproDir" || bail "Could not do: cd \"$ReproDir\""

msg "==============================================================================\nScenario $Scenario: $ScenarioTitle\n"
msg "\nPreliminary info: Show versions of p4/p4d on the PATH:"
cmd "p4 -V"
cmd "p4d -V"

msg "\nPreliminary setup: Spin up two local repos."
cmd "mkdir $ReproDir/Source"
cmd "mkdir $ReproDir/Target-1"
cmd "mkdir $ReproDir/Target-2"
cd "$ReproDir/Source" || bail "Could not do: cd \"$ReproDir/Source\""
msg "Operating in: $PWD"
cmd "p4 init -C0 -n"

# Sleep 1 second so the DVCS repos don't have the same ServerID.
sleep 1

cd "$ReproDir/Target-1" || bail "Could not do: cd \"$ReproDir/Target-1\""
msg "Operating in: $PWD"
cmd "p4 init -C0 -n"

# Sleep 1 second so the DVCS repos don't have the same ServerID.
sleep 1

cd "$ReproDir/Target-2" || bail "Could not do: cd \"$ReproDir/Target-2\""
msg "Operating in: $PWD"
cmd "p4 init -C0 -n"

cd "$ReproDir/Source" || bail "Could not do: cd \"$ReproDir/Source\""
msg "Operating in: $PWD"

msg "\\nAdd some files."
echo -e "// ONE_H\n#ifndef ONE_H\n#define ONE_H 1\n\n//Stuff goes here\n\n#endif //ONE_H\n" > One.h
cmd "p4 status"
msg "Reconcile and submit."
p4 rec && p4 submit -d "Added One.h."

cmd "p4 move -r One.h Two.h"
p4 submit -d "Moved One.h to Two.h."

cd ../Target-1 || bail "Could not do: cd ../Target-1"
msg "Operating in: $PWD"

msg "Creating this remote spec:"
cat "$AppHome/Everything.remote.p4s"

p4 -s remote -i < "$AppHome/Everything.remote.p4s" ||\
   errmsg "Failed to create remote spec 'Everything'."

msg "Attempting the fetch. This should succeed."

cmd "p4 fetch -r Everything" ||\
   errmsg "Fetch failed."

[[ "$ErrorCount" -eq 0 ]] || bail "Aborting repro due to unexpected errors."

msg "${H1}\\nOK, that's a regular fetch. Next, we'll mess with the history to create a partial rename situation."

cd ../Source || bail "Could not do: cd ../Source"
msg "Operating in: $PWD"

cmd "p4 -s obliterate -y One.h"
cmd "p4 describe -s 2"

cd ../Target-2 || bail "Could not do: cd ../Target-2"
msg "Operating in: $PWD"

msg "Creating this remote spec:"
cat "$AppHome/Everything.remote.p4s"

p4 -s remote -i < "$AppHome/Everything.remote.p4s" ||\
   errmsg "Failed to create remote spec 'Everything'."

[[ "$ErrorCount" -eq 0 ]] || bail "Aborting repro due to unexpected errors."

cmd "p4 fetch -r Everything" ||\
   errmsg "The fetch of the partial renamed failed (as expected, but not as desired)."

msg "${H1}\\nThe Result:"
if [[ "$ErrorCount" -eq 1 ]]; then
   msg "The above fetch failure is expected due to the current limitation on fetching partial renames."
   msg "Desired behavior would be to import the move/add action and leave behind the obliterated move/delete half of the action (possibly requiring a '-f' or '-F' force option). There may be limitations to that, but there should be a way to force through the fetch if we decide to ignore that specific type of fetch failure."
elif [[ "$ErrorCount" -eq 0 ]]; then
   msg "Yay, the fetch appears to have processed the fix!"
else
   bail "An unexpected nmber of errors ($ErrorCount) were encountered; exactly 1 was expected."
fi

exit "$ErrorCount"
