#!/bin/bash
set -u

# For documetation, run this script with the '-man' option:
# opt_perforce_sdp_backup.sh -man

#==============================================================================
# Declarations an Environment

declare ThisScript="${0##*/}"
declare Version=2.1.3
declare ThisUser=
declare ThisHost=${HOSTNAME%%.*}
declare PerforcePackageBase="/opt/perforce"
declare SDPPackageBase="$PerforcePackageBase/helix-sdp"
declare BackupMount=
declare BackupBase=
declare BackupDir=
declare BackupSSLDir=
declare Cmd=
declare SDPRoot="${SDP_ROOT:-/p4}"
declare SDPCommon="${SDPRoot}/common"
declare SDPCommonBin="${SDPCommon}/bin"
declare SDPCommonLib="${SDPCommon}/lib"
declare SSLDir="${SDPRoot}/ssl"
declare SDPOwner=
declare SDPGroup=
declare InstanceDir=
declare InstanceBackupDir=
declare -a InstanceList
declare Instance=
declare InstanceBinDir=
declare InstanceBinBackupDir=
declare H1="=============================================================================="
declare H2="------------------------------------------------------------------------------"
declare -i Debug=${SDP_DEBUG:-0}
declare -i ErrorCount=0
declare -i WarningCount=0
declare -i InstanceCount=0
declare -i SilentMode=0
declare LogsDir="/var/log/${ThisScript%.sh}"
declare LogLink="${ThisScript%.sh}"
declare Log=

#==============================================================================
# Local Functions

function msg () { echo -e "$*"; }
function dbg () { [[ "$Debug" -eq 0 ]] || msg "DEBUG: $*"; }
function errmsg () { msg "\\nError: ${1:-Unknown Error}\\n"; ErrorCount+=1; }
function warnmsg () { msg "\\nWarning: ${1:-Unknown Warning}\\n"; WarningCount+=1; }
function bail () { errmsg "${1:-Unknown Error}"; exit "${2:-1}"; }

#------------------------------------------------------------------------------
# Function: usage (required function)
#
# Input:
# $1 - style, either -h (for short form) or -man (for man-page like format).
# The default is -h.
#
# $2 - error message (optional).  Specify this if usage() is called due to
# user error, in which case the given message displayed first, followed by the
# standard usage message (short or long depending on $1).  If displaying an
# error, usually $1 should be -h so that the longer usage message doesn't
# obscure the error message.
#
# Sample Usage:
# usage
# usage -h
# usage -man
# usage -h "Incorrect command line usage."
#------------------------------------------------------------------------------
function usage
{
   local style=${1:--h}
   local usageErrorMessage=${2:-}

   [[ -n "$usageErrorMessage" ]] && msg "\\n\\nUsage Error:\\n\\n$usageErrorMessage\\n\\n"

   msg "USAGE for $ThisScript v$Version:

$ThisScript [-d|-D]

or

$ThisScript [-h|-man|-V]
"
   if [[ $style == -man ]]; then
      msg "
DESCRIPTION:
	This script is intended to called by a systemd timer on systems
	that support systemd.

	There is an opt_perforce_sdp_backup.service for this; it can be
	reviewed by doing:
	
	\$ systemctl cat opt_perforce_sdp_backup.service

	This service is triggered by a systemd timer, which can be viewed by:
	\$ systemctl cat opt_perforce_sdp_backup.timer

	If operating on a system without systemd, then it is OK to call this
	directly, e.g. via crontab. It must execute as root.

	This script backups key P4 Server Deployment Package (SDP) files and
	directories.  It does not back up actual P4 Server application data.
	The job of this script is to ensure any SDP files that are stored on
	the OS root volume are backed up a data volume that is backed up. If
	no data volume is found, an extra copy of files is made on the OS root
	volume.

	The recover_opt_perforce_sdp_from_backup.sh script uses the backup
	structure created by this script to rebuild local elements of the SDP.

OPTIONS:
 -L <log>
	Specify the path to a log file, or the special value 'off' to disable
	logging.  By default, all output (stdout and stderr) goes to a log file
	pointed to by a symlink:

	$LogLink

	The symlink is for convenience. It refers to the log from the most recent
	run of the script.

	Each time this script is run, a new timestamped log is started, and
	the symlink updated to reference the new/latest log during startup.  Log
	files have timestamps that go to the second (or millisecond if needed)
	to differentiate logs.

	NOTE: This script is self-logging.  Output displayed on the screen is
	simultaneously captured in the log file. Using redirection operators like
	'> log' or '2>&1' are unnecessary, as is using 'tee' (though using 'tee'
	or redirects is safe and harmless).

 -d
	Display debug messages.

 -D
	Set extreme debugging verbosity using bash 'set -x' mode. Implies -d.

 -si
	Silent Mode.  No output is displayed to the terminal (except for usage errors
	on startup). Output is captured in the log.  The '-si' cannot be used with
	'-L off'.

HELP OPTIONS:
 -h	Display short help message.
 -man	Display man-style help message.
 -V	Display script name and version.

FILES:
	/etc/systemd/system/${ThisScript%.sh}.service
	/etc/systemd/system/${ThisScript%.sh}.timer
"
   fi

   exit 2
}

#==============================================================================
# SDP Library Functions

declare -a BashLibList=(logging.lib utils.lib)
if [[ -d "$SDPCommonLib" ]]; then
   for BashLib in "${BashLibList[@]}"; do
      dbg "Loading lib $BashLib with: source \"$SDPCommonLib/$BashLib\""
      # shellcheck disable=SC1090
      source "$SDPCommonLib/$BashLib" ||\
         bail "Failed to load bash lib [$SDPCommonLib/$BashLib]. Aborting."
   done
fi

#==============================================================================
# Command Line Processing

declare -i shiftArgs=0

set +u
while [[ $# -gt 0 ]]; do
   case "$1" in
      (-h) usage -h;;
      (-man) usage -man;;
      (-V|-version|--version) msg "$ThisScript version $Version"; exit 0;;
      (-L) Log="$2"; shiftArgs=1;;
      (-si) SilentMode=1;;
      (-d) Debug=1;;
      (-D) Debug=1; set -x;; # Use bash 'set -x' extreme debug mode.
      (-*) usage -h "Unknown option ($1).";;
      (*) usage -h "Unknown parameter ($1).";;
   esac

   # Shift (modify $#) the appropriate number of times.
   shift; while [[ $shiftArgs -gt 0 ]]; do
      [[ $# -eq 0 ]] && usage -h "Incorrect number of arguments."
      shiftArgs=$shiftArgs-1
      shift
   done
done
set -u

#==============================================================================
# Command Line Verification

[[ "$SilentMode" -eq 1 && "$Log" == off ]] && \
   usage -h "The '-si' option cannot be used with '-Log off'."

#==============================================================================
# Main Program

trap terminate EXIT SIGINT SIGTERM

[[ -n "$Log" ]] || Log="${LogsDir}/${ThisScript%.sh}.${ThisHost}.$(date +'%Y-%m-%d-%H%M%S').log"

if [[ "$Log" != off ]]; then
   # Set LogsDir to the directory containing $Log. While LogsDir is defined above,
   # we need to reset it here in case the user used the -L option.
   LogsDir="${Log%/*}"
   if [[ ! -d "$LogsDir" ]]; then
      mkdir -p "$LogsDir" || bail "Couldn't do: mkdir -p \"$LogsDir\""
   fi

   touch "$Log" || bail "Couldn't touch log file: $Log"

   # Redirect stdout and stderr to a log file.
   if [[ "$SilentMode" -eq 0 ]]; then
      exec > >(tee "$Log")
      exec 2>&1
   else
      exec >"$Log"
      exec 2>&1
   fi

   msg "${H1}\\nLog is: $Log"
fi


ThisUser=$(id -n -u)
msg "${H1}\\nStarted $ThisScript v$Version as $ThisUser@$ThisHost on $(date)."

msg "Starting Preflight Checks."

if [[ "$ThisUser" == root ]]; then
   dbg "Verified: Running as root user."
else
   errmsg "Run this as root, not user '$ThisUser'."
fi

SDPOwner="$(stat --format=%U "$SDPCommonBin")" ||\
   errmsg "Could not owner of $SDPCommonBin with: stat --format=%U \"$SDPCommonBin\""
SDPGroup="$(stat --format=%G "$SDPCommonBin")" ||\
   errmsg "Could not group of $SDPCommonBin with: stat --format=%G \"$SDPCommonBin\""

msg "SDP owner/group is $SDPOwner:$SDPGroup, based on owner/group of $SDPCommonBin."

# Find the mount point to backup. For P4 Server and standalone P4 Proxy machines,
# this will be the mount point for /p4/N/{cache,checkpoints,depots} directory.
# For standalone Proxy and Broker machines, there may only be a local OS root
# volume, in which case there's no place for this script to backup to.
if ProbePath=$(find_p4depots_probe_path "$SDPRoot"); then
   BackupMount=$(get_mount_point "$ProbePath") ||\
      errmsg "Failed to determine mount point for: $ProbePath"
   dbg "Detected backup mount point: $BackupMount"
else
   warnmsg "Could not identify a P4Depots-related path under $SDPRoot."
   BackupMount=/
fi

# If there is no mounted volume, then we calculate the BackupBase by using
# 'pwd -P' to resolve symlinks to the ProbePath directory, which will start with
# /p4 and use the shorter symlink path, e.g. /p4/1/depots/.  The ProbePath
# reliably ends with a '/' character.  Then, we strip the ProbePath from the
# right side of the ProbePath and append "/backup" to get the BackupBase. So if
# the ProbePath is /p4/1/depots/, then a 'pwd -P' in that directory may look like
# /hxdepots/p4/1/depots (even if /hxdepots isn't a real mount point).  The
# BackupBase should then be /hxdepots/backup. Even though /hxdepots would also be
# on the OS root volume in that case, a local backup is better than no backup.
# For any production deployment for P4 Server host, it really should be a mount
# point, but that's outside the scope of this  script to control.
if [[ "$BackupMount" == / ]]; then
   warnmsg "No mount point detected for P4Depots-related dir could be determined."

   BackupBase="$(cd "$ProbePath" || return 1; pwd -P)"
   dbg "BB1: BB=[$BackupBase]"
   dbg "BB1: PP=[$ProbePath]"
   if [[ -n "$BackupBase" ]]; then
      dbg "BB2A: PP=[$ProbePath]"
      dbg "BB2A: BB=[$BackupBase] (Before)"
      BackupBase="${BackupBase%"${ProbePath%/}"}/backup"
      dbg "BB2A: BB=[$BackupBase] (After)"
   else
      dbg "BB2B: BB=[$BackupBase]"
      BackupBase=/backup
   fi
else
   dbg "BB3: BB=[$BackupBase]"
   BackupBase="${BackupMount%/}/backup"
fi

BackupDir="$BackupBase/opt_perforce_helix-sdp.$ThisHost"
BackupSSLDir="$BackupDir/ssl"

msg "Backup Base:     $BackupBase"
msg "Backup Dir:      $BackupDir"
msg "Backup SSL Dir:  $BackupSSLDir"

if [[ -d "$SDPPackageBase" ]]; then
   dbg "Verified: $SDPPackageBase directory exists."
else
   errmsg "SDP OS Package Structure [$SDPPackageBase] does not exists. Use this script only if that structure exists."
fi

if [[ ! -d "$BackupBase" ]]; then
   msg "Creating initial Backup Base directory: $BackupBase"
   mkdir -p "$BackupBase" ||\
      errmsg "Could not do: mkdir -p \"$BackupBase\""
   chmod 750 "$BackupBase" ||\
      errmsg "Could not do: chmod 750 \"$BackupBase\""
fi

if [[ -d "$BackupBase" && -w "$BackupBase" ]]; then
   dbg "Verified: $BackupBase directory exists and is writable."
else
   errmsg "The BackupBase directory [$BackupBase] does not exist or is not writable."
fi

if [[ "$ErrorCount" -eq 0 ]]; then
   dbg "Preflight checks OK."
else
   bail "Aborting due to failed preflight checks."
fi

msg "${H2}\\nBacking up SDP Package Base Dir."

Cmd="rsync -av \"$SDPPackageBase/\" \"$BackupDir\""
msg "Running: $Cmd"
if eval "$Cmd"; then
   msg "SDP Package Backup to $BackupDir was successful."
else
   errmsg "SDP Package Backup to $BackupDir failed."
fi

if [[ -e "$SSLDir" ]]; then
   msg "${H2}\\nBacking up SSL directory."
   Cmd="rsync -av \"$SSLDir/\" \"$BackupSSLDir\""

   msg "Running: $Cmd"
   if eval "$Cmd"; then
      msg "Successfully backed up SDP SSL Directory to $BackupSSLDir."
   else
      msg "Failed to backup SDP SSL Directory to $BackupSSLDir."
   fi
else
   msg "Skipping backup of the SSL directory [$SSLDir] because it does not exist."
fi

msg "${H2}\\nBacking up $SDPRoot/<N>/bin \"Instance Bin\" dirs."
dbg "Finding Instance Bin dirs with: find \"$SDPRoot/\" -maxdepth 2 -type d -name bin -print"
while IFS= read -r InstanceBinDir; do
   Instance="${InstanceBinDir#"$SDPRoot"/}"
   Instance="${Instance%%/*}"
   InstanceList[InstanceCount]="$Instance"
   InstanceCount+=1

   InstanceBinBackupDir="$BackupDir/${InstanceBinDir#"$SDPRoot"/}"

   msg "Backing up Instance Bin Dir: $InstanceBinDir."

   mkdir -p "$InstanceBinBackupDir" ||\
      errmsg "Could not do: mkdir -p \"$InstanceBinBackupDir\""
   chown "$SDPOwner:$SDPGroup" "$InstanceBinBackupDir" ||\
      errmsg "Could not do: chown \"$SDPOwner:$SDPGroup\" \"$InstanceBinBackupDir\""
   chmod 750 "$InstanceBinBackupDir" ||\
      errmsg "Could not do: chmod 750 \"$InstanceBinBackupDir\""

   Cmd="rsync -a \"$InstanceBinDir/\" \"$InstanceBinBackupDir\""

   dbg "Running: $Cmd"
   if eval "$Cmd"; then
      msg "Successfully backed up Instance Bin dir: $InstanceBinDir"
   else
      errmsg "Failed to backup Instance Bin dir: $InstanceBinDir"
   fi
done < <(find "$SDPRoot/" -maxdepth 2 -type d -name bin -print)

msg "${H2}\\nBacking up .p4* and other files in $SDPRoot/<N> \"Instance\" dirs."
for Instance in "${InstanceList[@]}"; do
   InstanceDir="${SDPRoot}/$Instance"
   InstanceBackupDir="$BackupDir/${InstanceDir#"$SDPRoot"/}"
   msg "Backing up .p4* and other files in: $InstanceDir"

   Cmd="find \"$InstanceDir/\" -maxdepth 1 -type f -exec cp -pf {} \"$InstanceBackupDir/.\" \;"
   dbg "Backing up files in $InstanceDir with this command: $Cmd"

   if eval "$Cmd"; then
      msg "Successfully backed up Instance Bin dir: $InstanceBinDir"
   else
      errmsg "Failed to backup Instance Bin dir: $InstanceBinDir"
   fi
done

msg "${H2}\\nBacking up any files in SDP Root ($SDPRoot)."

Cmd="find \"$SDPRoot/\" -maxdepth 1 -type f -exec cp -pf {} \"$BackupDir/.\" \;"
dbg "Backing up files in $SDPRoot with this command: $Cmd"

if eval "$Cmd"; then
   msg "Successfully backed up files in: $SDPRoot"
else
   errmsg "Failed to backup files in: $SDPRoot"
fi

if [[ "$ErrorCount" -eq 0 && "$WarningCount" -eq 0 ]]; then
   msg "\\n${H2}\\nSummary: SUCCESS:  P4 SDP Backup completed OK. Backups are here: $BackupDir"
elif [[ "$ErrorCount" -eq 0 ]]; then
   msg "\\n${H2}\\nSummary: SUCCESS:  P4 SDP Backup completed, but there were $WarningCount warnings. Backups are here: $BackupDir"
else
   msg "\\n${H2}\\nSummary: ERRORS:  There were $ErrorCount errors encountered attempting to backup the P4 SDP."
fi

exit "$ErrorCount"
