#!/bin/bash
#==============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://swarm.workshop.perforce.com/projects/perforce-software-sdp/view/main/LICENSE
#------------------------------------------------------------------------------

# This script is intended to be run on a standby replica.  It essentially just
# calls 'p4 admin checkpoint -Z' to request a checkpoint and exits. The actual
# checkpoint is created on the next journal rotation on the master.
declare -i RequestNow=0

export SDP_INSTANCE="${SDP_INSTANCE:-Unset}"
export SDP_INSTANCE="${1:-$SDP_INSTANCE}"
if [[ "$SDP_INSTANCE" == Unset ]]; then 
   echo "Instance parameter not supplied." 
   echo "You must supply the Perforce instance as a parameter to this script." 
   exit 1 
fi

# The '-now' option is currently an undocumented feature, though it may be
# documented in a future release (pending a separate change to normalize command
# line processing to add '-h' and '-man' options across all key scripts).  If
# '-now' is specified, after requesting the checkpoint on the local replica/edge,
# the commit server is then contacted and a 'p4 admin journal' is attempted, thus
# starting the checkpoint process on the local edge immediately
[[ "${2:-}" == "-now" ]] && RequestNow=1

# shellcheck disable=SC1091
source /p4/common/bin/p4_vars "$SDP_INSTANCE"
# shellcheck disable=SC1091
source /p4/common/bin/backup_functions.sh
# shellcheck disable=SC2034
LOGFILE=$LOGS/request_checkpoint.log

######### Start of Script ##########
check_vars
set_vars
rotate_last_run_logs
log "Start $P4SERVER Replica Request Checkpoint"
check_uid
check_dirs

[[ "$P4MASTER_ID" == "$SERVERID" ]] && \
   die "Error: Do not run request_replica_checkpoint.sh on the commit server."

"$P4CBIN"/p4login
request_replica_checkpoint

if [[ "$RequestNow" -eq 1 ]]; then
   log "Requesting journal rotation on the commit server due to '-now'."
   # Use timeout here so we don't hang if there's a firewall block.
   timeout 15s "$P4CBIN"/p4login -p "$P4MASTERPORT" >> "$LOGFILE" 2>&1 ||\
      die "The '-now' option was specified, but could login to commit server with: p4login -p $P4MASTERPORT"
   log "Running: $P4BIN -p $P4MASTERPORT admin journal"
   $P4BIN -p $P4MASTERPORT admin journal >> "$LOGFILE" 2>&1
fi

log "End $P4SERVER Replica Checkpoint"
mail_log_file "$HOSTNAME $P4SERVER replica request checkpoint log."
