# ============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://swarm.workshop.perforce.com/projects/perforce-software-sdp/view/main/LICENSE
# ----------------------------------------------------------------------------

# tag::includeManual[]
<#
    .Synopsis
        Live_checkpoint.ps1 checkpoints the live database and creates offline checkpoint
        
    .Description
        Admin access is required.
        This will lock the database for the duration which can be hours for large
        repositories!
        
    .Parameter sdp-instance
        The specified instance to checkpoint
        
    .Example
        live_checkpoint.ps1 Master
        
    .Example
        live_checkpoint.ps1 1
#>
# end::includeManual[]

[CmdletBinding()]
param ([string]$SDPInstance = $(throw "SDPInstance parameter is required."))

Set-StrictMode -Version 2.0

# Source the SDP Functions shared between scripts
$SDPFunctionsPath = Split-Path -parent $MyInvocation.MyCommand.Path | Join-Path -childpath "SDP-Functions.ps1" 
. $SDPFunctionsPath

$global:ScriptName = "Live-checkpoint.ps1"
$global:ScriptTask = "Live-checkpoint"
$global:LogFileName = "checkpoint.log"

Parse-SDPConfigFile $MyInvocation.MyCommand.Path
Create-LogFile

$OrigPath = convert-path .
Set-Location -Path $global:LOGS_DIR

try {
    Invoke-P4Login
    Ensure-CheckpointNotRunning
    Get-CurrentJournalCounter
    Rotate-Logfiles
    Run-Checkpoint
    Recreate-OfflineDBFiles
    Remove-OldCheckpointsAndJournals
    Log-DiskSpace
    Remove-OldLogs
    Log "End ${global:SDP_INSTANCE_P4SERVICE_NAME} ${global:ScriptTask}"
    Send-Email "${env:computername} ${global:SDP_INSTANCE_P4SERVICE_NAME} ${global:ScriptTask} log."
    Set-Counter
    Signal-CheckpointComplete
    Write-Output "`r`n${global:ScriptTask} completed successfully - see ${global:logfile}"
}
Catch
{
    write-error $error[0].ScriptStackTrace
    LogException $_.Exception
    Send-Email "FAILED: ${env:computername} ${global:SDP_INSTANCE_P4SERVICE_NAME} ${global:ScriptTask} log."
    Write-Output "`r`nFAILED - ${global:ScriptTask} - see ${global:logfile}"
}

Set-Location -Path $OrigPath
