#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# ==============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://swarm.workshop.perforce.com/projects/perforce-software-sdp/view/main/LICENSE
# ------------------------------------------------------------------------------

# tag::includeManual[]
"""
NAME:
    CheckStreamNameFormat.py

DESCRIPTION:
    Trigger to only require stream names to follow a regex (on 'form-save' event)

    To install, add a line to your Perforce triggers table like the following:

        check-stream-name form-save stream "python /p4/common/bin/triggers/CheckStreamNameFormat.py -c config.yaml -p %serverport% -u perforce %formfile% "

    or (if server is standard SDP and has appropriate environment defaults for P4PORT and P4USER):

        check-stream-name form-save stream "python /p4/common/bin/triggers/CheckStreamNameFormat.py -c /p4/common/config/Workflow.yaml %formfile% "

    You may need to provide the full path to python executable, or edit the path to the trigger.

    Also, don't forget to make the file executable, and setup the YAML configuration file (specified by -c parameter above):

# ------------------------------------------
# config.yaml

# msg_new_folder_not_allowed: An array of lines for the message
# For legibility it is good to have the first line blank
msg_invalid_stream_name:
  - ""
  - "The format of your stream name is not valid."
  - "Only the following formats are allowed:"

# valid_stream_name_formats: An array of python regexes to apply to stream
# names. Quote all values. ".*" will allow all names.
valid_stream_name_formats:
  - "//streams/(dev|rel|main)_.*"
  - "//morestreams/(teamA|teamB)/(dev)_.*"

"""
# end::includeManual[]

# Python 2.7/3.3 compatibility.
from __future__ import print_function

import sys
import re
from WorkflowTriggers import WorkflowTrigger
import os

trigger_name = os.path.basename(os.path.splitext(__file__)[0])

class CheckStreamNameFormat(WorkflowTrigger):
    """See module doc string for details"""

    def __init__(self, *args, **kwargs):
        WorkflowTrigger.__init__(self, **kwargs)
        self.parse_args(__doc__, args)

    def add_parse_args(self, parser):
        """Specific args for this trigger - also calls super class to add common trigger args"""
        parser.add_argument('-c', '--config-file', default=None,
                            help="Configuration file for trigger. Default: CheckStreamNameFormat.yaml")
        parser.add_argument('formfile',
                            help="Formfile containing job definition - %%formfile%% argument from triggers entry.")
        super(CheckStreamNameFormat, self).add_parse_args(parser)

    def run(self):
        """Runs trigger"""
        try:
            self.logger.debug("%s: firing" % trigger_name)
            config = self.load_config(self.options.config_file)
            errors = []
            for k in "valid_stream_name_formats msg_invalid_stream_name".split():
                if k not in config:
                    errors.append("Config file %s missing definition for %s" %
                                  (self.options.config_file, k))
            pats = []
            if not errors:
                for r in config['valid_stream_name_formats']:
                    try:
                        pats.append(re.compile(r))
                    except:
                        errors.append("Invalid python regex: %s" % r)
            if errors:
                msg = "%s: Invalid config file for trigger %s\n" % (trigger_name, str(errors))
                self.message(msg)
                return 1
            self.setupP4()
            self.p4.connect()
            with open(self.options.formfile, 'r') as f:
                contents = f.read()
            stream = self.p4.parse_stream(contents)
            if "Date" in stream:   # If Date field set then abort because change has already been processed
                self.logger.debug("%s: Stream already processed so exiting" % trigger_name)
                return 0

            sname = stream["Stream"]
            for pat in pats:
                if pat.search(sname):
                    self.logger.debug("%s: stream valid: %s" % (trigger_name, sname))
                    return 0

            err_msg = "\n".join(config['msg_invalid_stream_name']) +\
                "\nStream: %s" % sname
            err_msg += "\nValid formats:\n" + "\n\t".join(config['valid_stream_name_formats'])
            self.message(err_msg)
            self.logger.warning(err_msg)
            return 1

        except Exception:
            return self.reportException()

        return 0

if __name__ == '__main__':
    """ Main Program"""
    trigger = CheckStreamNameFormat(*sys.argv[1:])
    sys.exit(trigger.run())