#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# ==============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://swarm.workshop.perforce.com/projects/perforce-software-sdp/view/main/LICENSE
# ------------------------------------------------------------------------------

# tag::includeManual[]
"""
NAME:
    CreateSwarmReview.py

DESCRIPTION:
    This trigger is intended for use with Swarm installations

    It will search for any jobs associated with the changelist and find any reviews
    associated with that job.
    If found it will update the review with this changelist
    Otherwise it will create a new Swarm review with a template description
    and with configurable reviewers as requested.

    To install, add a line to your Perforce triggers table like the following:

        create_swarm_review change-commit //... "python /p4/common/bin/triggers/CreateSwarmReview.py -c Workflow.yaml -p %serverport% -u perforce %change% "

    or (if server is standard SDP and has appropriate environment defaults for P4PORT and P4USER):

        create_swarm_review change-commit //... "python /p4/common/bin/triggers/CreateSwarmReview.py -c Workflow.yaml %change% "

    Note that -c/--config specifies a yaml file - see Workflow.yaml for example.

    You may need to provide the full path to python executable, or edit the path to the trigger.

    Also, don't forget to make the file executable.
"""
# end::includeManual[]

# Python 2.7/3.3 compatibility.
from __future__ import print_function

import sys

import WorkflowTriggers
import requests
import os

trigger_name = os.path.basename(os.path.splitext(__file__)[0])

class CreateSwarmReview(WorkflowTriggers.WorkflowTrigger):
    """See module doc string for details"""

    def __init__(self, *args, **kwargs):
        WorkflowTriggers.WorkflowTrigger.__init__(self, **kwargs)
        self.parse_args(__doc__, args)

    def add_parse_args(self, parser):
        """Specific args for this trigger - also calls super class to add common trigger args"""
        parser.add_argument('-c', '--config-file', default=None,
                            help="Configuration file for trigger. Default: CreateSwarmReview.yaml")
        parser.add_argument('--test-mode', default=False, action='store_true',
                            help="Trigger runs in test mode - for unit testing only")
        parser.add_argument('change', help="Change to process - %%change%% argument from triggers entry.")
        super(CreateSwarmReview, self).add_parse_args(parser)

    def run(self):
        """Runs trigger"""
        try:
            self.logger.debug("%s: trigger firing" % trigger_name)
            self.setupP4()
            self.p4.connect()
            change = self.getChange(self.options.change)

            config = self.load_config(self.options.config_file)
            if not self.project_flag_true(config, change, 'post_submit_create_review'):
                self.logger.debug("%s: no project found" % trigger_name)
                return 0

            swarm_url = self.get_swarm_url()
            base_url = "%s%s" % (swarm_url, config['api'])
            auth = (config['user'], config['ticket'])

            # Look for jobs attached to this changelist and get all their changelists
            fixes = self.p4.run_fixes("-c", self.options.change)
            other_changes = []
            job_names = []
            jobs = []
            if fixes and len(fixes) > 0:
                for f in fixes:
                    if f['Job'] not in job_names:
                        job_names.append(f['Job'])
                total_fixes = []
                for j in job_names:
                    total_fixes.extend(self.p4.run_fixes("-j", j))
                    jobs.append(self.p4.fetch_job(j))
                # Create list of unique changes
                for f in total_fixes:
                    if f['Change'] != self.options.change and f['Change'] not in other_changes:
                        other_changes.append(f['Change'])

            # Now get any reviews for those changes, and pick the last one (first in array)
            review_to_update = None
            if self.project_flag_true(config, change, 'update_review') and other_changes:
                chg_search = ["change[]=%s" % c for c in other_changes]
                url = '%s/reviews?%s' % (base_url, "&".join(chg_search))
                self.logger.debug("Get: %s" % (url))
                resp = requests.get(url, auth=auth)
                resp_json = resp.json()
                self.logger.debug("Result: %s" % (resp_json))
                if 'reviews' in resp_json:
                    reviews =  [str(r['id']) for r in resp_json['reviews']]
                    self.logger.debug("Reviews: %s" % (reviews))
                    if len(reviews) > 1:
                        self.logger.warning("Multiple reviews found")
                    if reviews and len(reviews) > 0:
                        review_to_update = reviews[0]

            # Now create new or update existing review
            if review_to_update:
                url = '%s/reviews/%s/changes/' % (base_url, review_to_update)
                data = {'change': self.options.change, 'id': review_to_update}
                self.logger.debug("Post: %s data: %s" % (url, data))
                resp = requests.post(url, auth=auth, data=data)
                self.logger.debug("Result: %s" % (resp.json()))
            else:
                url = '%s/reviews' % base_url
                job_desc = ""
                change_desc = change.desc.rstrip("\n")
                if jobs and len(jobs) > 0:
                    job_desc = jobs[0]['Description']
                    job_desc = job_desc.rstrip("\n")
                description = self.formatReviewDescription(config['review_description'],
                                                           changeDescription=change_desc,
                                                           jobDescription=job_desc)
                # Need to make a list of tuples due to strange way Swarm requires array of reviewers
                data = [('change', self.options.change), ('description', description)]
                prj = self.get_project(config, change)
                if 'default_reviewers' in prj:
                    for resp in prj['default_reviewers']:
                        data.append(('reviewers[]', resp))
                self.logger.debug("Post: %s data: %s" % (url, data))
                resp = requests.post(url, auth=auth, data=data)
                resp_json = resp.json()
                self.logger.debug("Result: %s" % (resp_json))
                # Now we update the author of the review - unfortunately can't do it in just one API call
                self.logger.debug("Result: %s" % (resp_json))
                review_id = str(resp_json['review']['id'])
                url = '%s/reviews/%s' % (base_url, review_id)
                data = [('author', change.user)]
                self.logger.debug("Patch: %s data: %s" % (url, data))
                resp = requests.patch(url, auth=auth, data=data)
                self.logger.debug("Result: %s" % (resp.json()))
        except Exception:
            return self.reportException()

        return 0


if __name__ == '__main__':
    """ Main Program"""
    trigger = CreateSwarmReview(*sys.argv[1:])
    sys.exit(trigger.run())
