
# Usage:
# cd /dir/in/your/workspace/containing/this/file
# source ./env.sh

export APP_HOME=$(dirname $PWD)
export TOOLS_DIR=$APP_HOME/tools
export WORKSHOP_PROJECT=perforce-software-sdp
export WORKSHOP_PROJECT_TAG=sdp
export P4U_HOME="${APP_HOME}/dev/Server/Unix/p4/common/bin"
export P4U_LIB="${APP_HOME}/dev/Server/Unix/p4/common/lib"

# Set WORKSHOP_USER in your own personal shell environment
# to avoid guessing.
export WORKSHOP_USER=${WORKSHOP_USER:-Unset}
export PATH="$PATH:$TOOLS_DIR:$P4U_HOME"

# Alias to list open SDP jobs.
alias ojobs='bash $TOOLS_DIR/sdp_jobs_report.sh'
alias ofjobs='bash $TOOLS_DIR/sdp_jobs_report.sh -st open,blocked,inprogress,fixed'
alias ajobs='bash $TOOLS_DIR/sdp_jobs_report.sh -a'
alias mjobs='bash $TOOLS_DIR/sdp_jobs_report.sh -me'

#------------------------------------------------------------------------------
# Function fjob ($job1 $job2 ...)
# Change job status to fixed. Job can look like SDP-### or ###.
function fjob () {
   local job=
   local jobProj=
   for job in $*; do
      # If the job looks like 323, prefix with 'SDP-'
      [[ "$job" =~ ^[0-9]+$ ]] && job="${WORKSHOP_PROJECT_TAG^^}-$job"

      if [[ ! "$job" =~ ^SDP- ]]; then
         echo -e "\\nError: Not changing status of non-SDP job [$job]."
         continue
      fi

      jobProj=$(p4 -ztag -F %Project% job -o "$job")

      if [[ "$jobProj" != "$WORKSHOP_PROJECT" ]]; then
         if [[ "$jobProj" == "setme" ]]; then
            echo -e "\\nError: Job [$job] does not exist. Skipping it."
         else
            echo -e "\\nError: Job [$job] has wrong project [$jobProj], expected [$WORKSHOP_PROJECT]. Skipping it."
         fi
         continue
      fi

      # Try to change the status to fixed.
      p4 --field Status=fixed job -o "$job" | p4 job -i
   done
}

#------------------------------------------------------------------------------
# Function: guess_workshop_user ($user)
#
# Try to guess the workshop user based on the current OS user.  The assumption
# is that local OS user will NOT be the same as the userid on The Workshop, due
# to intentionally differing user naming conentions. The guessing logic can be
# avoided by defining WORKSHOP_USER environment variable in your personal shell
# environment to your P4USER value for public.perforce.com:1666.
#
# This is strictly a convenience function for SDP project members.
#------------------------------------------------------------------------------
function guess_workshop_user () {
   declare user="${1:-$USER}"
   declare workshopUser=

   if [[ "$WORKSHOP_USER" == "Unset" ]]; then
      case "$user" in
         (rjackson) workshopUser="russell_jackson";;
         (rcowham) workshopUser="robert_cowham";;
         (ttyler) workshopUser="tom_tyler";;
         (mzinthefer) workshopUser="mark_zinthefer";;
         (amorriss) workshopUser="amo";;
         (*) workshopUser="$USER";;
      esac
   else
      workshopUser="$USER"
   fi
   echo $workshopUser
}

#------------------------------------------------------------------------------
# Usage: njob [type] [description headline, quotes optional]
# The default for 'type' is Feature; valid values are Bug/Doc/Feature/Problem.
#
# Auto spelling correct logic is keyed off the first letter, case-insensitive:
#
# * 'B' is 'Bug'
# * 'D' is 'Doc'
# * 'F' is 'Feature'
# * 'P' is 'Problem'
#
# Examples:
# njob
# njob b
# njob B This is my job headline
# njob FEatuRE "This is my job headline"
# njob f This is my cool new feature.

function njob () {
   declare type=${1:-Feature}
   if [[ $# -lt 2 ]]; then
      echo -e "\\nUsage: njob {f|b} Your description goes here.\\n"
      return 1
   fi
   shift
   declare desc=${*:-__EDITME_DESC__}
   declare user=$(guess_workshop_user ${USER:-unknown})
   declare -i colonHandling=0
   declare -i i=1
   declare newJobFile=new_${i}.job
   declare -a ComponentList
   declare -a ComponentDescList
   declare Component
   declare ComponentDesc
   declare -i ComponentCount=0
   declare -i ComponentValid
   declare tmpFile=

   # Simple auto-correct.  I'd like to use the bash built-in ${var^^} syntax,
   # but that requires bash 4.x, which isn't standard on Mac yet (but is
   # ubiquitous on modern Linux).
   type=$(echo $type | awk '{print tolower($0)}')
   if [[ ${type} == "b"* ]]; then
      type=Bug
   elif [[ ${type} == "d"* ]]; then
      type=Doc
   elif [[ ${type} == "p"* ]]; then
      type=Problem
   elif [[ ${type} == "f"* ]]; then
      type=Feature
   else
      echo -e "\\nWarning: Type value specified [$type] should be B (Bug), F (Feature), D (Doc), or P (Problem).  Unless the Public Depot server admins have updated the list of supported values.\\n"
   fi

   if [[ -r $TOOLS_DIR/components.txt ]]; then
      # Load components list
      while read data; do
         [[ $data == "#"* ]] && continue
         Component=$data
         Component=${Component%%:*}
         ComponentDesc=$data
         ComponentDesc=${ComponentDesc##*:}
         ComponentList[$ComponentCount]=$Component
         ComponentDescList[$ComponentCount]=$ComponentDesc
         ComponentCount+=1
      done < $TOOLS_DIR/components.txt

      ComponentCount=0
      echo -e "SDP Components:"
      for Component in ${ComponentList[*]}; do
         echo "$((ComponentCount+1)). $Component: ${ComponentDescList[$ComponentCount]}"
         ComponentCount+=1
      done

      ComponentValid=0
      while [[ $ComponentValid -eq 0 ]]; do
         echo -e -n "\\nSelect a Component (1-$ComponentCount): "
         read ComponentId
         if [[ -n "$ComponentId" ]]; then
            ComponentId=$((ComponentId-1))
            Component=${ComponentList[$ComponentId]}
            for ccheck in ${ComponentList[*]}; do
               [[ "$Component" == "$ccheck" ]] && ComponentValid=1
            done
         else
            Component
            ComponentValid=1
         fi
      done
   fi

   # Get a unique job file name in the current dir by incrementing an
   # integer.
   while [[ -f $newJobFile ]]; do
      i=$((i+1))
      newJobFile=new_${i}.job
   done

   echo -e "\\nGenerating new job file: $newJobFile:\\n"

   template=$TOOLS_DIR/template.job.p4s

   if [[ $desc == *":"* ]]; then
      colonHandling=1
      desc=$(echo $desc | sed "s/:/__COLON__/g")
   fi

   sed -e "s:__EDITME_TYPE__:$type:g" \
      -e "s:__EDITME_USER__:$user:g" \
      -e "s:__EDITME_PROJECT__:$WORKSHOP_PROJECT:g" \
      -e "s:__EDITME_COMPONENT__:$Component:g" \
      -e "s:__EDITME_DESC__:$desc:g" $template > $newJobFile

   if [[ $colonHandling -eq 1 ]]; then
      tmpFile=tmp.$newJobFile.$$.$RANDOM
      sed "s/__COLON__/:/g" $newJobFile > $tmpFile
      mv -f "$tmpFile" $newJobFile
   fi

   cat $newJobFile
   echo -e "\\n"
}

# Display a job with 'less', sans the big comment block.
function jo () {
   p4 job -o $1 | grep -v "^#" | less
}

# Alias for 'Get Job Header' script.
alias gj='gjh.sh'
alias bp='/p4/tools/bp.sh'
