#!/bin/bash
#==============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://workshop.perforce.com/view/p4-sdp/main/LICENSE
#------------------------------------------------------------------------------

#------------------------------------------------------------------------------
# Version ID Block. Relies on +k filetype modifier.
# VersionID='$Id: //p4-sdp/dev_rebrand/Server/Unix/p4/common/bin/refresh_P4ROOT_from_offline_db.sh#3 $ $Change: 31617 $'

declare -i PreserveOldP4ROOT=0

export SDP_INSTANCE=${SDP_INSTANCE:-Undefined} 
export SDP_INSTANCE=${1:-$SDP_INSTANCE} 
if [[ $SDP_INSTANCE == Undefined ]]; then 
   echo "Instance parameter not supplied." 
   echo "You must supply the Perforce instance as a parameter to this script." 
   exit 1 
fi 

# The '-K' option is currently an undocumented feature, though it may be documented
# in a future release (pending a separate change to normalize command line processing
# to add '-h' and '-man' options across all key scripts).  If '-K' is specified, this
# avoids removing the formerly saved db.* files in the old P4ROOT.
[[ "${2:-}" == "-K" ]] && PreserveOldP4ROOT=1

# shellcheck disable=SC1091
source /p4/common/bin/p4_vars "$SDP_INSTANCE"
# shellcheck disable=SC1091
source /p4/common/bin/backup_functions.sh
LOGFILE="$LOGS/refresh_P4ROOT_from_offline_db.log"

######### Start of Script ##########
check_vars
set_vars
rotate_last_run_logs
log "Start $P4SERVER Refresh P4ROOT from offline_db."
check_uid
check_dirs
check_offline_db_usable
if ! "$P4BIN" info -s > /dev/null 2>&1; then
   die "Error: p4d is not running as expected. Aborting."
fi
ckp_running
"$P4CBIN"/p4login
get_journalnum
get_offline_journal_num
stop_p4d
p4d_truncate_journal
replay_journals_to_offline_db

# On an edge or replica server, we have to replay the active journal as well because
# edges/replicas have been running since the journal was rotated.
if [[ "$SERVERID" != "$P4MASTER_ID" ]]; then
   replay_active_journal_to_offline_db
fi

# If run on a 'standby' replica, artificially set state to match statejcopy, as we have
# just successfully replayed the entire content of the locally pulled journal into the
# offline_db.
if [[ "$STANDBYSERVER" -eq 1 ]]; then
   if [[ -r "$P4ROOT/statejcopy" ]]; then
      log "Setting $P4ROOT/state to $P4ROOT/statejcopy"
      cp -f "$P4ROOT/statejcopy" "$P4ROOT/state"
   else
      log "Warning: Expected $P4ROOT/statejcopy file not found."
   fi
fi

# Call copy_readonly_clients_dir() to determine if clients.readonly.dir needs
# to be copied to offline_db.
if copy_readonly_clients_dir; then
   log "Successfully handled client.readonly.dir data (if any)."
else
   start_p4d
   die "Aborting due to failure to copy the client.readonly.dir. Because this failure mode did not affect data used by p4d, it was safe to continue using the p4d service, so the p4d service was restarted prior this abort."
fi

switch_db_files
start_p4d

if [[ "$PreserveOldP4ROOT" -eq 0 ]]; then
   log "Removing db files from save directory since we know the journal successfully replayed at this point."
   # The offline_db directory was the root, so this is where the save directory is now.
   rm -f "${OFFLINE_DB}"/save/db.* >> "$LOGFILE" 2>&1
else
   SavedDBDir="${OFFLINE_DB}/moved.$(date +'%Y-%m-%d-%H%M')"
   log "Preserving former P4ROOT files in $SavedDBDir due to -K."
   mkdir -p "$SavedDBDir" ||\
      die "Could not do: mkdir -p \"$SavedDBDir\""
   mv "${OFFLINE_DB}"/save/db.* "${SavedDBDir}/." ||\
      die "Could not do: mv \"${OFFLINE_DB}\"/save/db.* \"${SavedDBDir}/.\""
fi

recreate_offline_db_files
get_offline_journal_num
replay_journals_to_offline_db
# shellcheck disable=SC2034
ROOTDIR=$OFFLINE_DB

# If on the master, kick off the standard offline checkpoint.  For servers other than
# the master, skip this.
if [[ "$SERVERID" == "$P4MASTER_ID" ]]; then
   dump_checkpoint
fi
remove_old_checkpoints_and_journals
check_disk_space
remove_old_logs
set_counter
ckp_complete

log "End $P4SERVER Refresh P4ROOT from offline_db."
mail_log_file "$HOSTNAME $P4SERVER Refresh P4ROOT from offline_db log."
