# -*- encoding: UTF8 -*-
# Test harness for depot_verify_chunks.py

from __future__ import print_function

import sys
import unittest
import os
import re

import P4
curr_dir = os.path.dirname(os.path.abspath(__file__))
sys.path.insert(0, os.path.join(curr_dir, "..", "..", "bin", "triggers", "tests"))

from p4testutils import TestCase, P4Server, localDirectory, create_file, append_to_file

from depot_verify_chunks import depot_verify_chunks, parse_human_fmt

os.environ["LOGS"] = "."
LOGGER_NAME = "Testdepot_verify_chunks"
LOG_FILE = "log-Testdepot_verify_chunks.log"


class Testdepot_verify_chunks(TestCase):
    def __init__(self, methodName='runTest'):
        super(Testdepot_verify_chunks, self).__init__(LOGGER_NAME, LOG_FILE, methodName=methodName)

    def setUp(self):
        self.server = P4Server()
        p4 = self.server.p4
        self.p4 = p4
        p4.logger = self.logger

    def tearDown(self):
        pass

    def testParse(self):
        """Check parsing of args"""
        valid_args = {'1': 1, '1.9': 1, '1k': 1024, '1.1m': int(1024 * 1024 * 1.1),
                      '2k': 2 * 1024, '.5k': 512, '.5K': 512,
                      '1G': 1024 * 1024 * 1024}
        for k in valid_args.keys():
            exp = valid_args[k]
            act = parse_human_fmt(k)
            self.assertEqual(exp, act, "Expected %s got %d" % (exp, act))

        invalid_args = ['a', ' 1.', '1.9c', '1.2.3']
        for k in invalid_args:
            try:
                act = parse_human_fmt(k)
                self.assertFalse(True, "Expected assertion error for %s" % (k))
            except TypeError:
                pass

    def testdepot_verify_chunks(self):
        """check that fixes are appropriate controlled"""

        p4 = self.p4

        files = localDirectory(self.server.client_root, "files")
        d1 = localDirectory(files, "d1")
        d2 = localDirectory(files, "d2")
        for i in range(10):
            f = os.path.join(d1, "file_%d" % i)
            create_file(f, "0123456789\n" * 10)
            f = os.path.join(d2, "file_%d" % i)
            create_file(f, "0123456789\n" * 10)
        p4.run_add("%s/..." % files)
        p4.run_submit("-d", "Initial change")

        dvc = depot_verify_chunks("-p", p4.port, "-u", p4.user, "-m", "1000000", "//depot/...")
        results = dvc.run()
        self.assertEqual(['//depot/...'], results)

        dvc = depot_verify_chunks("-p", p4.port, "-u", p4.user, "-m", "1000", "//depot/...")
        results = dvc.run()
        self.assertEqual(['//depot/files/d1/*', '//depot/files/*', '//depot/files/d2/*', '//depot/*'], results)

        dvc = depot_verify_chunks("-p", p4.port, "-u", p4.user, "-m", "1500", "//depot/...")
        results = dvc.run()
        self.assertEqual(['//depot/files/d1/...', '//depot/files/*', '//depot/files/d2/...', '//depot/*'], results)

        p4.run_populate("//depot/files/...", "//depot/dev/...")
        dvc = depot_verify_chunks("-p", p4.port, "-u", p4.user, "-m", "1500", "//depot/...")
        results = dvc.run()
        self.assertEqual(['//depot/dev/...', '//depot/*', '//depot/files/d1/...',
                          '//depot/files/*', '//depot/files/d2/...'], results)

if __name__ == '__main__':
    unittest.main()
