#!/bin/bash
#==============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://workshop.perforce.com/view/p4-sdp/main/LICENSE
#------------------------------------------------------------------------------

#------------------------------------------------------------------------------
# Version ID Block. Relies on +k filetype modifier.
# VersionID='$Id: //p4-sdp/dev_rebrand/Server/Unix/p4/common/bin/replica_status.sh#3 $ $Change: 31617 $'

export SDP_INSTANCE=${SDP_INSTANCE:-Undefined}
export SDP_INSTANCE=${1:-$SDP_INSTANCE}

if [[ "$SDP_INSTANCE" == Undefined ]]; then
   echo "Instance parameter not supplied."
   echo "You must supply the Perforce instance as a parameter to this script."
   exit 1
fi

declare ErrorText=""
declare -i ExitCode=0
declare Cmd=

# shellcheck disable=SC1091
source /p4/common/bin/p4_vars "$SDP_INSTANCE"
# shellcheck disable=SC1091
source /p4/common/bin/backup_functions.sh

LOGFILE="$LOGS/replica_status.log"

rotate_log_file "$LOGFILE"

"$P4CBIN/p4login" >> "$LOGFILE" 2>&1

P4="$P4BIN -p ${P4PORT} -u $P4USER"

echo -e "\\nMetadata replication check." > "$LOGFILE"

# shellcheck disable=SC2072
if [[ "$P4D_VERSION" > "2021.1" ]]; then
   Cmd="$P4 -s pull -ljv"
else
   Cmd="$P4 -s pull -lj"
fi

echo -e "$Cmd" >> "$LOGFILE" 2>&1
$Cmd >> "$LOGFILE" 2>&1

if ( grep -q "Connection refused" "$LOGFILE" ); then
   ErrorText="ERROR - The replica is offline!"
   echo -e "\\n$ErrorText\\n" >> "$LOGFILE"
   ExitCode=1
fi

if ( grep -q "\-1" "$LOGFILE" ); then
   ErrorText="ERROR - The replica is out of sync!"
   echo -e "\\n$ErrorText\\n" >> "$LOGFILE"
   ExitCode=1
fi

if ( grep -qi "passw" "$LOGFILE" ); then
   ErrorText="ERROR - The admin user is not logged in!"
   echo -e "\\n$ErrorText\\n" >> "$LOGFILE"
   ExitCode=1
fi

if [[ -r "$P4ROOT/db.config" && -n "$SERVERID" ]]; then
   # shellcheck disable=SC2154
   copy_jd_table "db.config" "$P4ROOT"
   LbrReplication=$("$P4DBIN" -r "$JDTmpDir" -cshow | grep "${SERVERID}: lbr.replication" | cut -d ' ' -f 4)
   remove_jd_tables

   if [[ "$LbrReplication" == "readonly" ]]; then
      # shellcheck disable=SC2129
      echo -e "\\nArchive file replication check." >> "$LOGFILE"
      echo "$P4 -s pull -l" >> "$LOGFILE" 2>&1
      $P4 -s pull -l >> "$LOGFILE" 2>&1

      if ( grep -qE 'Transfer of librarian file .* failed' "$LOGFILE" ); then
         ErrorText="ERROR - Replica archive transfers have failed!"
         echo -e "\\n$ErrorText\\n" >> "$LOGFILE"
         ExitCode=1
      fi

      Cmd="$P4 -s pull -ls"
      echo "$Cmd" >> "$LOGFILE" 2>&1
      $Cmd >> "$LOGFILE" 2>&1
   else
      echo -e "\\nSkipping archive replication check for metadata-only replica." >> "$LOGFILE"
   fi
fi

mail_log_file "$ErrorText $HOSTNAME $P4SERVER Daily Replica Status."

exit "$ExitCode"

