#!/bin/bash
set -u

#==============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://workshop.perforce.com/view/p4-sdp/main/LICENSE
#------------------------------------------------------------------------------

#==============================================================================
# Declarations and Environment

# Version ID Block. Relies on +k filetype modifier.
#------------------------------------------------------------------------------
# shellcheck disable=SC2016
declare VersionID='$Id: //p4-sdp/dev_rebrand/Server/Unix/p4/common/bin/templates/template.sh#4 $ $Change: 31912 $'
declare VersionStream=${VersionID#*//}; VersionStream=${VersionStream#*/}; VersionStream=${VersionStream%%/*};
declare VersionCL=${VersionID##*: }; VersionCL=${VersionCL%% *}
declare Version=${VersionStream}.${VersionCL}
[[ "$VersionStream" == r* ]] || Version="${Version^^}"

declare ThisScript=${0##*/}
declare CmdLine="$0 $*"
declare ThisUser=
declare ThisHost=${HOSTNAME%%.*}
declare -i ErrorCount=0
declare -i WarningCount=0
declare -i SilentMode=0
declare -i Debug=0
declare -i NoOp=1
declare -i i=0
declare Log=
declare OldLogTimestamp=
declare LogTimestamp=
declare LogLink=
declare H1="=============================================================================="
declare H2="------------------------------------------------------------------------------"
declare SDPInstance=
declare SDPRoot=${SDP_ROOT:-/p4}

# Color support.
declare GREEN=
declare RED=
declare YELLOW=
declare RESET=

#==============================================================================
# Local Functions
function msg () { echo -e "$*"; }
function msg_green  { msg "${GREEN}$*${RESET}"; }
function msg_yellow { msg "${YELLOW}$*${RESET}"; }
function msg_red    { msg "${RED}$*${RESET}"; }
function dbg () { [[ "$Debug" -eq 0 ]] || msg "DEBUG: $*"; }
function errmsg () { msg_red "\\nError: ${1:-Unknown Error}\\n"; ErrorCount+=1; }
function warnmsg () { msg_yellow "\\nWarning: ${1:-Unknown Warning}\\n"; WarningCount+=1; }
function bail () { errmsg "${1:-Unknown Error}"; exit "$ErrorCount"; }

declare ThisUser=
declare ThisHost=${HOSTNAME%%.*}
declare -i ErrorCount=0
declare -i Debug=${DEBUG:-0}
declare -i SilentMode=0
declare -i WarningCount=0
declare Log=
declare H1="=============================================================================="
declare H2="------------------------------------------------------------------------------"
declare SDPInstance=${SDP_INSTANCE:-}
declare SDPRoot=${SDP_ROOT:-/p4}
declare SDPInstanceVars=

declare -i NO_OP=0
declare -i VERBOSITY=3

#------------------------------------------------------------------------------
# Function: terminate
# shellcheck disable=SC2317
function terminate
{
   # Disable signal trapping.
   trap - EXIT SIGINT SIGTERM

   dbg "$ThisScript: EXITCODE: $ErrorCount"

   # Stop logging.
   [[ "$Log" == off ]] || msg "\\nLog is: $Log\\n${H1}"

   # With the trap removed, exit with the error count.
   exit "$ErrorCount"
}

#------------------------------------------------------------------------------
# Function: usage (required function)
#
# Input:
# $1 - style, either -h (for short form) or -man (for man-page like format).
# The default is -h.
#
# $2 - error message (optional).  Specify this if usage() is called due to
# user error, in which case the given message displayed first, followed by the
# standard usage message (short or long depending on $1).  If displaying an
# error, usually $1 should be -h so that the longer usage message doesn't
# obscure the error message.
#
# Sample Usage:
# usage
# usage -h
# usage -man
# usage -h "Incorrect command line usage."
#------------------------------------------------------------------------------
function usage
{
   declare style=${1:--h}
   declare usageErrorMessage=${2:-Unset}

   if [[ "$usageErrorMessage" != Unset ]]; then
      msg "\\n\\nUsage Error:\\n\\n$usageErrorMessage\\n\\n"
   fi

   msg "USAGE for $ThisScript version $Version:

$ThisScript EDITME <required> literal [-optional] [-i <sdp_instance>] [-L <log>] [-si] [-y] [-d|-D]

or

$ThisScript [-h|-man|-V]
"
   if [[ $style == -man ]]; then
      msg "
DESCRIPTION: EDITME

OPTIONS:
 -i <sdp_instance>
	Specify the SDP Instance.  If not specified, the \$SDP_INSTANCE variable is
	referenced, otherwise the default is '1'.

 -L <log>
	Specify the path to a log file, or the special value 'off' to disable
	logging.  By default, all output (stdout and stderr) goes to
	EDITME_DEFAULT_LOG

	NOTE: This script is self-logging.  That is, output displayed on the screen
	is simultaneously captured in the log file.  Do not run this script with
	redirection operators like '> log' or '2>&1', and do not use 'tee.'

 -si	Operate silently.  All output (stdout and stderr) is redirected to the log
	only; no output appears on the terminal.  This cannot be used with
	'-L off'.
      
	EDITME: This is useful when running from cron, as it prevents automatic
	email from being sent by cron directly, as it does when a script called
	from cron generates any output.  This script is then responsible for
	email handling, if any is to be done.

 -n	No-Op.  Prints commands instead of running them.

 -D     Set extreme debugging verbosity.

HELP OPTIONS:
 -h	Display short help message
 -man	Display man-style help message
 -V	Display version info for this script and its libraries.

FILES:

EXAMPLES:

SEE ALSO:
"
   fi

   exit 2
}

#==============================================================================
# Command Line Processing

declare SampleArgument=""; ### EDITME
declare -i shiftArgs=0

set +u
while [[ $# -gt 0 ]]; do
   case $1 in
      (-s) SampleArgument="$2"; shiftArgs=1;; ### EDITME
      (-h) usage -h;;
      (-man) usage -man;;
      (-V) msg "$ThisScript version $Version"; exit 0;;
      (-L) Log="$2"; shiftArgs=1;;
      (-si) SilentMode=1;;
      (-y) NoOp=0;;
      (-d) Debug=1;;
      (-D) set -x;; # Debug; use 'set -x' mode.
      (-*) usage -h "Unknown option ($1).";;
      (*)
         if [[ -z "$SDPInstance" ]]; then
            SDPInstance="$1"
         else
            usage -h "Unknown parameter '$1'. SDP instance already set as '$SDPInstance'."
         fi
      ;;
   esac

   # Shift (modify $#) the appropriate number of times.
   shift; while [[ $shiftArgs -gt 0 ]]; do
      [[ $# -eq 0 ]] && usage -h "Incorrect number of arguments."
      shiftArgs=$shiftArgs-1
      shift
   done
done
set -u

source "$SDPRoot/common/bin/p4_vars" "$SDPInstance" ||\
   bail "Could not do: source \"$SDPRoot/common/bin/p4_vars\" \"$SDPInstance\""

declare BASH_LIBS="$P4CLIB/librun.sh"

for bash_lib in $BASH_LIBS; do
   # shellcheck disable=SC1090
   source "$bash_lib" ||\
      bail "Could not do: source \"$bash_lib\""
done

#==============================================================================
# Command Line Verification

[[ "$SilentMode" -eq 1 && "$Log" == off ]] && \
   usage -h "Cannot use '-si' with '-L off'."

[[ -n "$Log" ]] || \
   Log="${LOGS:-${HOME:-/tmp}}/${ThisScript%.sh}.$(date +'%Y%m%d-%H%M%S').log"

[[ -z "$SDPInstance" ]] && SDPInstance="${SDP_INSTANCE:-}"
[[ -z "$SDPInstance" ]] && usage -h "The SDP instance parameter is required unless SDP_INSTANCE is set. To set SDP_INSTANCE, do:\\n\\tsource $SDPRoot/common/bin/p4_vars INSTANCE\\n\\nreplacing INSTANCE with your SDP instance name."

#==============================================================================
# Main Program

# shellcheck disable=SC1090 disable=SC1091
source "$SDPRoot"/common/bin/p4_vars "$SDPInstance" ||\
   bail "Could not do: source /p4/common/bin/p4_vars \"$SDPInstance\""
# shellcheck disable=SC1090 disable=SC1091
source "$SDPRoot/common/bin/log_functions.sh" ||\
   bail "Could not do: source \"$SDPRoot/common/bin/log_functions.sh\""

trap terminate EXIT SIGINT SIGTERM

# Detect support for colors
if [[ $SilentMode -eq 0 ]] \
  && command -v tput >/dev/null 2>&1 \
  && [[ -t 1 ]] \
  && [[ "$(tput colors)" -ge 8 ]]; then
  RED="$(tput setaf 1)" 
  GREEN="$(tput setaf 2)"
  YELLOW="$(tput setaf 3)"
  RESET="$(tput sgr0)"
else
  RED=; GREEN=; YELLOW=; RESET=
fi

# If the user specifies a log file file with '-L', write to the specified file.
# If no log was specified, create a default log file using a timestamp in the
# LOGS dir, and immediately update the symlink for the default log to point to
# it.
if [[ "$Log" != off ]]; then
   # If $Log is not yet defined, set it to a reasonable default.
   if [[ -z "$Log" ]]; then
      LogTimestamp=$(date +'%Y-%m-%d-%H%M%S')
      Log="${LOGS:-~}/${ThisScript%.sh}.${LogTimestamp}.log"
      # Make sure we have a unique log file. Prefer a human-readable timestamp
      # using hours/minutes/seconds. Append milliseconds if needed to ensure
      # a unique filename.
      while [[ -e "$Log" ]]; do
         LogTimestamp=$(date +'%Y-%m-%d-%H%M%S.%3N')
         Log="$LOGS/${ThisScript%.sh}.${LogTimestamp}.$i.log"
         i+=1
      done
   fi

   # The LogLink symlink has no timestamp. It points to the most recent log file.
   LogLink="${LOGS:-~}/${ThisScript%.sh}.log"

   if [[ -e "$LogLink" ]]; then
      if [[ -L "$LogLink" ]]; then
         rm -f "$LogLink"
      else
         # If the name that should be a symlink is not a symlink, move it aside before
         # creating the symlink.
         OldLogTimestamp=$(get_old_log_timestamp "$LogLink")
         mv -f "$LogLink" "${LogLink%.log}.${OldLogTimestamp}.log" ||\
            bail "Could not move old log file aside; tried: mv -f \"$LogLink\" \"${LogLink%.log}.${OldLogTimestamp}.log\""
      fi
   fi

   # Initialize the log file.
   touch "$Log" || bail "Couldn't touch log file [$Log]."

   # Use a subshell to set the LogLink so the 'cd' doesn't persist.
   ( cd "$LOGS"; ln -s "${Log##*/}" "${LogLink##*/}"; ) ||\
       bail "Couldn't initialize log symlink; tried: ln -s \"$Log\" \"$LogLink\""

   # Redirect stdout and stderr to a log file.
   if [[ "$SilentMode" -eq 0 ]]; then
      if [[ -n "$GREEN" ]]; then
         exec > >( tee \
            >(sed -r \
            -e 's/\x1B\[[0-9;]*[a-zA-Z]//g' \
            -e 's/\x1B\(B//g' >>"$Log"))
      else
         exec > >(tee "$Log")
      fi
      exec 2>&1
   else
      exec >"$Log"
      exec 2>&1
   fi

   msg "${H1}\\nLog is: $Log"
fi

ThisUser=$(id -n -u)
msg "Starting $ThisScript version $Version as $ThisUser@$ThisHost on $(date) as:\\n$CmdLine\\n"

if [[ "$NoOp" -eq 0 ]]; then
   msg "LIVE operation mode."
else
   msg "This is a DRY RUN."
fi

run "ls /tmp" "List temp area."

msg "This is a standard message."
dbg "This is a debug message, only displayed if -d (or -D) is used."

run "ls" "Sample Arg specified with '-s <arg>' is [$SampleArgument].  Directory List:" 1

run "/bin/sleep 1" "Taking a short nap." 1

# Illustrate stringing together a series of commands using a '|'.
# run can handle this.
run "ls -lrth ${LOGS:-~}/|grep \" $P4USER \" |head -2|tail -1" "Showing last file:" 0

warnmsg "Sample WarningMsg 1"
errmsg "Sample Error Message 2"
warnmsg "Sample WarningMsg 3"
errmsg "Sample Error Message 4"

#rrun localhost "sleep 2\\nls -lrt /tmp/" "Doing an ls on simulated remote host." 1 1 0
#
#rrun localhost "ls -l /tmp/*" "Grepping remote output for /tmp/hello" 0 0 "/tmp/hello" ||\
#   warnmsg "/tmp/hello was not found in remote output."

if [[ "$ErrorCount" -eq 0 && "$WarningCount" -eq 0 ]]; then
   msg_green "${H2}\\nAll processing completed successfully.\\n"
elif [[ "$ErrorCount" -eq 0 ]]; then
   msg_yellow "${H2}\\nProcessing completed with no errors, but with $WarningCount warnings.  Review above output carefully.\\n" 
else
   msg_yellow "${H2}\\nProcessing completed with $ErrorCount errors and $WarningCount warnings.  Review above output carefully.\\n" 
fi

# Illustrate using $SECONDS to display runtime of a script.
msg_green "Time: $ThisScript took $((SECONDS/3600)) hours $((SECONDS%3600/60)) minutes $((SECONDS%60)) seconds.\\n"

# See the terminate() function, where this script really exits.
exit "$ErrorCount"
