# ============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://workshop.perforce.com/view/p4-sdp/main/LICENSE
# ----------------------------------------------------------------------------

# tag::includeManual[]
<#
    .Synopsis
        Creates a filtered edge checkpoint from a master offline database.
        The resulting checkpoint can be copied to the remote edge server
        and restored using recover-edge.ps1
        
    .Description
        Create filtered checkpoint using the server spec (output of 
        'p4 server -o', and in particular the fields RevisionDataFilter:
        and  ArchiveDataFilter: which specify filtering).
        
        IMPORTANT NOTE:
        Because this uses the offline database, you can not just edit an
        existing server spec to change the filter and have it picked up by
        this script.
        
        After changing the server spec for the live server, you must ensure that the metadata
        changes are reflected in the offline_db - which is easy to do by
        running daily-backup.ps1 as normal (this rotates the journal and
        applies it to offline_db).
        
        Alternatively you can wait until the following day to run this
        script, by which time a scheduled daily backup should have run!
        
        Output of this script:
        A gzipped (filtered) checkpoint file, reflecting the name of the
        current instance. E.g. if current instance is Master, then result
        will be c:\p4\Master\checkpoints\p4_Master.ckp.filtered-edge.193.gz
        where 193 is the latest numbered checkpoint in that directory.
        
    .Parameter SDPInstance
        The specified instance to process, e.g. 1 or Master
        
    .Parameter EdgeServer
        The specified id of edge server (a server spec visible in output of
        'p4 servers' command).
        
    .Example
        create-filtered-edge-checkpoint.ps1 Master Edge-server
        
#>
# end::includeManual[]

[CmdletBinding()]
param ([string]$SDPInstance = $(throw "SDPInstance parameter is required."),
       [string]$EdgeServer = $(throw "EdgeServer parameter is required."))

Set-StrictMode -Version 2.0

# Source the SDP Functions shared between scripts
$SDPFunctionsPath = Split-Path -parent $MyInvocation.MyCommand.Path | Join-Path -childpath "SDP-Functions.ps1" 
. $SDPFunctionsPath

$global:ScriptName = "create-filtered-edge-checkpoint.ps1"
$global:ScriptTask = "Create Filtered edge checkpoint from master"
$global:LogFileName = "create-filtered-edge-checkpoint.log"

Parse-SDPConfigFile $MyInvocation.MyCommand.Path
Create-LogFile

$OrigPath = convert-path .
Set-Location -Path $global:LOGS_DIR

try {
    $checkpoint_prefix = -join($global:CHECKPOINTS_DIR, "\", $global:SDP_INSTANCE_P4SERVICE_NAME, ".ckp.")
    $checkpoint_path = -join($checkpoint_prefix, "[0-9]*.gz")
    $files = @(Get-ChildItem -Path $checkpoint_path | Sort-Object -Property LastWriteTime -Descending)
    if (!$files -or $files.count -eq 0) {
        $journal = Get-JournalCounter $global:OFFLINE_DB_DIR
    } else {
        $journal = $files[0].Name | select-string '\.ckp\.([0-9]*)\.gz' | % {$_.Matches} | % {$_.Groups[1].Value}
    }
    $CkpFile = -join($checkpoint_prefix, "filtered-edge.", $journal, ".gz")
    $md5file = -join($checkpoint_prefix, "filtered-edge.", $journal, ".md5")
    remove-files $CkpFile 0
    remove-files $md5file 0
    
    log "Checkpointing from master offline_db skipping tables not used on the edge - with filtering."
    $cmd = "$global:P4DEXE -r $global:OFFLINE_DB_DIR -K ""$global:EXCLUDED_EDGE_TABLES"" -P $EdgeServer -jd -z $CkpFile"
    run-cmd-with-check $cmd "ERROR - attempting to create filtered checkpoint"
    
    Log "End ${global:SDP_INSTANCE_P4SERVICE_NAME} ${global:ScriptTask}"
    Write-Output "`r`n${global:ScriptTask} completed successfully - see ${global:logfile}"
    Log "Please copy $CkpFile to remote server and restore using recover-edge.ps1"
}
Catch
{
    write-error $error[0].ScriptStackTrace
    LogException $_.Exception
    Write-Output "`r`nFAILED - ${global:ScriptTask} - see ${global:logfile}"
}

Set-Location -Path $OrigPath
