# ============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://workshop.perforce.com/view/p4-sdp/main/LICENSE
# ----------------------------------------------------------------------------

# tag::includeManual[]
<#
    .Synopsis
        Daily_Backup.ps1 performs journal rotation to offline database and creates offline checkpoint
        
    .Description
        Admin access is required.
        Also recovers from the offline checkpoint to ensure that it is good, and that offline
        DB files are unfragmented.
        
    .Parameter sdp-instance
        The specified instance to backup
        
    .Example
        daily_backup.ps1 Master
        
    .Example
        daily_backup.ps1 1
#>
# end::includeManual[]

[CmdletBinding()]
param ([string]$SDPInstance = $(throw "SDPInstance parameter is required."))

Set-StrictMode -Version 2.0

# Source the SDP Functions shared between scripts
$SDPFunctionsPath = Split-Path -parent $MyInvocation.MyCommand.Path | Join-Path -childpath "SDP-Functions.ps1" 
. $SDPFunctionsPath

$global:ScriptName = "Daily-backup.ps1"
$global:ScriptTask = "Daily Backup"
$global:LogFileName = "checkpoint.log"

Parse-SDPConfigFile $MyInvocation.MyCommand.Path
Create-LogFile

$OrigPath = convert-path .
Set-Location -Path $global:LOGS_DIR

try {
    Invoke-P4Login
    Check-OfflineDBUsable
    Check-OfflineDBExists
    Ensure-CheckpointNotRunning
    Get-CurrentJournalCounter
    Get-OfflineJournalCounter
    Truncate-Journal
    Rotate-Logfiles
    Replay-JournalsToOfflineDB
    Create-OfflineCheckpoint
    Recreate-OfflineDBFiles
    Remove-OldCheckpointsAndJournals
    Log-DiskSpace
    Remove-OldLogs
    Log "End ${global:SDP_INSTANCE_P4SERVICE_NAME} ${global:ScriptTask}"
    Send-Email "${env:computername} ${global:SDP_INSTANCE_P4SERVICE_NAME} ${global:ScriptTask} log."
    Set-Counter
    Signal-CheckpointComplete
    Write-Output "`r`n${global:ScriptTask} completed successfully - see ${global:logfile}"
}
Catch
{
    write-error $error[0].ScriptStackTrace
    LogException $_.Exception
    Send-Email "FAILED: ${env:computername} ${global:SDP_INSTANCE_P4SERVICE_NAME} ${global:ScriptTask} log."
    Write-Output "`r`nFAILED - ${global:ScriptTask} - see ${global:logfile}"
}

Set-Location -Path $OrigPath
