# ============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://workshop.perforce.com/view/p4-sdp/main/LICENSE
# ----------------------------------------------------------------------------

# tag::includeManual[]
<#
    .Synopsis
        rotate-log-files.ps1 rotates key log files for service. 
        
    .Description
        Rotates all log files found and zips them.
        Useful for replicas which may not otherwise have scheduled tasks set.
        For use in Windows Task Scheduler.
        
    .Parameter sdp-instance
        The specified SDP instance to verify
        
    .Example
        rotate-log-files.ps1 Master
        
    .Example
        rotate-log-files.ps1 1
#>
# end::includeManual[]

[CmdletBinding()]
param ([string]$SDPInstance = $(throw "SDPInstance parameter is required."))

Set-StrictMode -Version 2.0

# Source the SDP Functions shared between scripts
$SDPFunctionsPath = Split-Path -parent $MyInvocation.MyCommand.Path | Join-Path -childpath "SDP-Functions.ps1" 
. $SDPFunctionsPath

$global:ScriptName = "rotate-log-files.ps1"
$global:ScriptTask = "Rotate Log Files"
$global:LogFileName = "rotate-log-files.log"

Parse-SDPConfigFile $MyInvocation.MyCommand.Path
Create-LogFile

$OrigPath = convert-path .
Set-Location -Path $global:LOGS_DIR

try {
    Invoke-P4Login
    Get-CurrentJournalCounter
    Rotate-LogFiles
    Remove-OldLogs
    Log "End ${global:SDP_INSTANCE_P4SERVICE_NAME} ${global:ScriptTask}"
    # Don't bother emailing on success
    Write-Output "`r`n${global:ScriptTask} completed successfully - see ${global:logfile}"
}
Catch
{
    write-error $error[0].ScriptStackTrace
    LogException $_.Exception
    Send-Email "FAILED: ${env:computername} ${global:SDP_INSTANCE_P4SERVICE_NAME} ${global:ScriptTask} log."
    Write-Output "`r`nFAILED - ${global:ScriptTask} - see ${global:logfile}"
}

Set-Location -Path $OrigPath
