# ============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://workshop.perforce.com/view/p4-sdp/main/LICENSE
# ----------------------------------------------------------------------------

# tag::includeManual[]
<#
    .Synopsis
        sync-replica.ps1 copies checkpoint files from master to replica to ensure
        that they are local in case of replica reseeding being required.
        
    .Description
        Admin access is required.
		This script rebuilds the offline_db from the copied most recent checkpoint.
        
    .Parameter sdp-instance
        The specified instance to process
        
    .Example
        sync-replica.ps1 Master
        
    .Example
        sync-replica.ps1 1
#>
# end::includeManual[]

[CmdletBinding()]
param ([string]$SDPInstance = $(throw "SDPInstance parameter is required."))

Set-StrictMode -Version 2.0

# Source the SDP Functions shared between scripts
$SDPFunctionsPath = Split-Path -parent $MyInvocation.MyCommand.Path | Join-Path -childpath "SDP-Functions.ps1" 
. $SDPFunctionsPath

$global:ScriptName = "sync-replica.ps1"
$global:ScriptTask = "Sync Replica"
$global:LogFileName = "sync-replica.log"

Parse-SDPConfigFile $MyInvocation.MyCommand.Path
Create-LogFile

if (!(Test-IsAdmin)) {
    write-warning "Please run this script with admin privileges"
    break
}

$OrigPath = convert-path .
Set-Location -Path $global:LOGS_DIR

Function Invoke-P4LoginToParent () {
    Log "Logging in to parent P4 Instance"
    $cmd = "$global:P4exe -p $global:P4TARGET -u $global:SDP_P4SUPERUSER login -a < $global:ADMIN_PASS_FILE >> $global:LogFile 2>&1"
    Log $cmd
    cmd /c $cmd
    if ($lastexitcode -ne 0) {
        throw "ERROR - failed to login to parent!"
    }
}

try {
    Check-OfflineDBUsable
    Check-OfflineDBExists
    Invoke-P4LoginToParent

    $cmd = "$global:P4exe -p $global:P4TARGET -u $global:SDP_P4SUPERUSER counter journal"
    $result = run-cmd-get-output $cmd "Error - failed to get parent journal counter!"
    [int]$MasterJournalNum = $result
    [int]$global:JOURNAL_NUM = $MasterJournalNum - 1        
    Log "Parent journalcounters ${MasterJournalNum}/${$global:JOURNAL_NUM}"

    $cmd = "xcopy /D /I ${global:REMOTE_CHECKPOINTS_DIR}\*.* ${global:CHECKPOINTS_DIR}\*.*"
    run-cmd $cmd "Failed to xcopy checkpoints"

    Ensure-CheckpointNotRunning
	
    # The following call uses $global:JOURNAL_NUM as setup above.
    Recreate-OfflineDBFiles
    Rotate-Logfiles
    Remove-OldCheckpointsAndJournals
    Log-DiskSpace
    Remove-OldLogs
    Log "End ${global:SDP_INSTANCE_P4SERVICE_NAME} ${global:ScriptTask}"
    Send-Email "${env:computername} ${global:SDP_INSTANCE_P4SERVICE_NAME} ${global:ScriptTask} log."
    Signal-CheckpointComplete
    Write-Output "`r`n${global:ScriptTask} completed successfully - see ${global:logfile}"
}
Catch
{
    write-error $error[0].ScriptStackTrace
    LogException $_.Exception
    Send-Email "FAILED: ${env:computername} ${global:SDP_INSTANCE_P4SERVICE_NAME} ${global:ScriptTask} log."
    Write-Output "`r`nFAILED - ${global:ScriptTask} - see ${global:logfile}"
}

Set-Location -Path $OrigPath
