#!/usr/bin/env python

"""
This script will email users that their clients haven't been accessed in the number of weeks defined
in the weeks variable, and warn them that it will be deleted in one week if they do not use it.
"""
# Python 2.7/3.3 compatibility.
from __future__ import print_function

import os
import re
import time
import smtplib
import traceback
import sys
import string
import accessdates
import platform
import sdputils

if len(sys.argv) > 1:
  SDP_INSTANCE = str(sys.argv[1])
else:
  SDP_INSTANCE = '1'

utils = sdputils.SDPUtils(SDP_INSTANCE)
p4 = utils.p4
server = utils.server
utils.login()

##########################################################################
#####                                #####
#####  CONFIGURATION VARIABLES: Modify in maintenance.cfg as needed. #####
#####                                #####

weeks1warn = int(utils.get('weeks1warn'))
weeks2warn = int(utils.get('weeks2warn'))
administrator = utils.get('administrator')
complain_from = utils.get('complain_from')
cc_admin = (utils.get('ccadmin'))
mailhost = utils.get('mailhost')
repeat = int(utils.get('repeat'))
sleeptime = int(utils.get('sleeptime'))

message = """
Your workspace, %s will be unloaded in %s unless you use it to get the latest
revision of a file. The sync will update the access time, and your
client will not be unloaded.

If your workspace is unloaded, you can reload it with the following command:

p4 reload -c %s

If you need to keep this workspace active:
Just login to Perforce using the workspace, right click any file, and choose the
option "Get Latest Revision".

Thanks for your cooperation,
The Perforce Admin Team
"""

def complain(mailport, complaint):
  '''
  Send a plaintive message to the human looking after this script if we
  have any difficulties.  If no email address for such a human is given,
  send the complaint to stderr.
  '''
  complaint = complaint + '\n'
  if administrator:
    mailport.sendmail(complain_from, [administrator],
              'Subject: Perforce Email Client Deletion Review Daemon Problem\n\n' + complaint)
  else:
    sys.stderr.write(complaint)


def mailit(mailport, sender, recipient, message):
  '''
  Try to mail message from sender to the user using SMTP object
  mailport.  complain() if there are any problems.
  '''

  recipients = []
  recipients.append(recipient)
  if cc_admin:
    recipients.append(cc_admin)

  try:
    failed = mailport.sendmail(sender, recipients, message)
  except:
    exc_type, exc_value, exc_tb = sys.exc_info()
    failed = traceback.format_exception(exc_type, exc_value, exc_tb)

  if failed:
    complain(mailport, 'The following errors occurred:\n\n' +
         repr(failed) +
         '\n\nwhile trying to email from\n' +
         repr(sender) + '\nto ' +
         repr(recipient) + '\nwith body\n\n' + message)


def mailwarning(client, warningtime, mailport):
  os.system('%s client -o "%s" > client.txt' % (p4, client))
  clientfile = open("client.txt", "r")
  for clientline in clientfile.readlines():
    m = re.search(r"^Owner:\s(.*)\s", clientline)
    if m:
      owner = m.group(1)
      os.system('%s user -o "%s" > owner.txt' % (p4, owner))
      ownerfile = open("owner.txt", "r")
      email = owner
      for ownerline in ownerfile.readlines():
        m2 = re.search("^Email:\s(.*)\s", ownerline)
        if m2:
          email = m2.group(1)
      ownerfile.close()
      os.remove("owner.txt")
      subject = "Subject: Client %s on %s server is scheduled for deletion." % (client, server)
      messagebody = 'From: ' + administrator + '\n' + \
              'To: ' + email + '\n' + subject + '\n' + (message % (client, warningtime, client))
      mailit(mailport, administrator, email, messagebody)
  clientfile.close()
  os.remove("client.txt")


def warnusers(weeks1warn, weeks2warn, mailport):
  clients1warn = []
  clients2warn = []

  accessdates.createlist("clients", "client", weeks2warn)
  clientlist = open("clients.txt", "r")
  for client in clientlist.readlines():
    client = client.rstrip()
    client = re.sub("\$", "\\\$", client)
    clients2warn.append(client)
  clientlist.close()
  os.remove("clients.txt")

  accessdates.createlist("clients", "client", weeks1warn)
  clientlist = open("clients.txt", "r")
  for client in clientlist.readlines():
    client = client.rstrip()
    client = re.sub("\$", "\\\$", client)
    if client not in clients2warn:
      clients1warn.append(client)
  clientlist.close()
  os.remove("clients.txt")

  for client in clients1warn:
    mailwarning(client, "two weeks", mailport)
  for client in clients2warn:
    mailwarning(client, "one week", mailport)


def loop_body(mailhost):
  # Note: there's a try: wrapped around everything so that the program won't
  # halt.  Unfortunately, as a result you don't get the full traceback.
  # If you're debugging this script, remove the special exception handlers
  # to get the real traceback, or figure out how to get a real traceback,
  # by importing the traceback module and defining a file object that
  # will take the output of traceback.print_exc(file=mailfileobject)
  # and mail it (see the example in cgi.py)
  try:
    mailport = smtplib.SMTP(mailhost)
  except:
    sys.stderr.write('Unable to connect to SMTP host "' + mailhost +
             '"!\nWill try again in ' + repr(sleeptime) +
             ' seconds.\n')
  else:
    try:
      warnusers(weeks1warn, weeks2warn, mailport)
    except:
      exc_type, exc_value, exc_tb = sys.exc_info()
      complain(mailport, 'Client Deletion Review daemon problem:\n\n%s' % \
           traceback.format_exception(exc_type, exc_value, exc_tb))
    try:
      mailport.quit()
    except:
      sys.stderr.write('Error while doing SMTP quit command (ignore).\n')


def main():
  while (repeat):
    loop_body(mailhost)
    time.sleep(sleeptime)
  else:
    loop_body(mailhost)


###############################################################################
# main
if __name__ == '__main__':
  main()
