#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# ==============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://workshop.perforce.com/view/p4-sdp/main/LICENSE
# ------------------------------------------------------------------------------

# tag::includeManual[]
"""
NAME:
    DefaultSwarmReviewDesc.py

DESCRIPTION:
    Create a default change description (form-in) for swarm user - so updates Swarm review changelist

    Can be used together with CheckChangeDesc.py (change-submit)

    To install, add a line to your Perforce triggers table like the following:

        default-swarm-desc form-in change "python /p4/common/bin/triggers/DefaultSwarmReviewDesc.py -p %serverport% -u perforce %user% %formfile% "

    or (if server is standard SDP and has appropriate environment defaults for P4PORT and P4USER):

        default-swarm-desc form-in change "python /p4/common/bin/triggers/DefaultSwarmReviewDesc.py %user% %formfile% "

    You may need to provide the full path to python executable, or edit the path to the trigger.
"""
# end::includeManual[]

# Python 2.7/3.3 compatibility.
from __future__ import print_function

import sys
import WorkflowTriggers
import shutil
import os

trigger_name = os.path.basename(os.path.splitext(__file__)[0])

class DefaultSwarmReviewDesc(WorkflowTriggers.WorkflowTrigger):
    """See module doc string for details"""

    def __init__(self, *args, **kwargs):
        WorkflowTriggers.WorkflowTrigger.__init__(self, **kwargs)
        self.parse_args(__doc__, args)

    def add_parse_args(self, parser):
        """Specific args for this trigger - also calls super class to add common trigger args"""
        parser.add_argument('-c', '--config-file', default=None,
                            help="Configuration file for trigger. Default: CreateSwarmReview.yaml")
        parser.add_argument('user', help="User carrying out the command - %%user%% argument from triggers entry.")
        parser.add_argument('formfile',
                            help="Formfile containing job definition - %%formfile%% argument from triggers entry.")
        super(DefaultSwarmReviewDesc, self).add_parse_args(parser)

    def run(self):
        """Runs trigger"""
        try:
            self.logger.debug("%s: firing" % trigger_name)
            config = self.load_config(self.options.config_file)
            if not 'review_description' in config or not config['review_description']:
                self.logger.debug("%s: no review_description found" % trigger_name)
                return 0
            self.setupP4()
            self.p4.connect()
            with open(self.options.formfile, 'r') as f:
                contents = f.read()
            chg = self.p4.parse_change(contents)
            if "Date" in chg:   # If Date field set then abort because change has already been processed
                self.logger.debug("%s: Change already processed so exiting" % trigger_name)
                return 0
            if not "User" in chg or chg["User"] != config['swarm_user']:
                self.logger.debug("%s: Ignoring change as not Swarm user" % trigger_name)
                return 0

            # job_desc = ""
            # if jobs and len(jobs) > 0:
            #     job_desc = jobs[0]['Description']
            #     job_desc = job_desc.rstrip("\n")
            description = self.formatReviewDescription(config['review_description'])
            chg["Description"] = description
            contents = self.p4.format_change(chg)
            self.logger.debug("Updated chg: %s" % str(chg))
            tempfile = "%s~" % self.options.formfile
            shutil.move(self.options.formfile, tempfile)
            with open(self.options.formfile, 'w') as f:
                f.write(contents)
            os.remove(tempfile)
        except Exception:
            return self.reportException()

        return 0


if __name__ == '__main__':
    """ Main Program"""
    trigger = DefaultSwarmReviewDesc(*sys.argv[1:])
    sys.exit(trigger.run())
