#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# ==============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://workshop.perforce.com/view/p4-sdp/main/LICENSE
# ------------------------------------------------------------------------------

# tag::includeManual[]
"""
NAME:
    ControlStreamCreation.py

DESCRIPTION:
    Trigger to only allow certain users/groups to create streams (form-save)

    To install, add a line to your Perforce triggers table like the following:

        control-stream-create form-save stream "python /p4/common/bin/triggers/ControlStreamCreation.py -c config.yaml -p %serverport% -u perforce %user% %formfile% "

    or (if server is standard SDP and has appropriate environment defaults for P4PORT and P4USER):

        control-stream-create form-save stream "python /p4/common/bin/triggers/ControlStreamCreation.py -c /p4/common/config/Workflow.yaml %user% %formfile% "

    You may need to provide the full path to python executable, or edit the path to the trigger.

    Also, don't forget to make the file executable, and setup the YAML configuration file (specified by -c parameter above):

# ------------------------------------------
# config.yaml

# msg_new_folder_not_allowed: An array of lines for the message
# For legibility it is good to have the first line blank
msg_cant_create_stream:
  - ""
  - "You are not allowed to create new streams."
  - "Check with the admins to be added to an appropriate group."

# can_create_streams: An array of projects with user or group names who are allowed
# to create streams matching path regexes.
#   name: Project name
#   stream_paths: An array of python regex patterns
#       to which the check is applied - should be quoted. Use ".*" to match everything
can_create_streams:
  - name:       Name of this project
    stream_paths:
      - "//my_streams_depot/.*"
    allowed_users_groups:
      - user1
      - group1
      - group2

  - name:       Project A
    stream_paths:
      - "//depot2/.*"
    allowed_users_groups:
      - user2
      - group3

"""
# end::includeManual[]

# Python 2.7/3.3 compatibility.
from __future__ import print_function

import sys
from WorkflowTriggers import WorkflowTrigger, GroupMemberChecker
import os
import re

trigger_name = os.path.basename(os.path.splitext(__file__)[0])

class ControlStreamCreation(WorkflowTrigger):
    """See module doc string for details"""

    def __init__(self, *args, **kwargs):
        WorkflowTrigger.__init__(self, **kwargs)
        self.parse_args(__doc__, args)

    def add_parse_args(self, parser):
        """Specific args for this trigger - also calls super class to add common trigger args"""
        parser.add_argument('-c', '--config-file', default=None,
                            help="Configuration file for trigger. Default: ControlStreamCreation.yaml")
        parser.add_argument('user', help="User carrying out the command - %%user%% argument from triggers entry.")
        parser.add_argument('formfile',
                            help="Formfile containing job definition - %%formfile%% argument from triggers entry.")
        super(ControlStreamCreation, self).add_parse_args(parser)

    def get_stream_project(self, config, stream_name):
        for prj in config['can_create_streams']:
            if not 'name' in prj or not 'stream_paths' in prj:
                continue
            pats = []
            for r in prj['stream_paths']:
                try:
                    pats.append(re.compile(r))
                except:
                    self.logger.error("Invalid python regex: %s in prj %s" % (r, str(prj)))
            for pat in pats:
                if pat.search(stream_name):
                    self.logger.debug("%s: stream matched: %s" % (trigger_name, stream_name))
                    return prj
            return {}

    def run(self):
        """Runs trigger"""
        try:
            self.logger.debug("%s: firing" % trigger_name)
            config = self.load_config(self.options.config_file)
            errors = []
            for k in "can_create_streams msg_cant_create_stream".split():
                if k not in config:
                    errors.append("Config file %s missing definition for %s" %
                                  (self.options.config_file, k))
            if errors:
                msg = "%s: Invalid config file for trigger %s\n" % (trigger_name, str(errors))
                self.message(msg)
                return 1
            self.setupP4()
            self.p4.connect()
            with open(self.options.formfile, 'r') as f:
                contents = f.read()
            stream = self.p4.parse_stream(contents)
            if "Date" in stream:   # If Date field set then abort because stream has already been processed
                self.logger.debug("%s: Stream already processed so exiting" % trigger_name)
                return 0

            # Check for stream name matching.
            prj = self.get_stream_project(config, stream["Stream"])
            if not prj:
                self.logger.debug("%s: Stream not found in config: %s" % (trigger_name, stream["Stream"]))
                return 0

            self.logger.debug("%s: prj: %s" % (trigger_name, str(prj)))
            if 'allowed_users_groups' not in prj:
                self.logger.debug("%s: No field allowed_users_groups found" % (trigger_name))
                return 0

            # Now check for users and groups who are allowed to create streams
            gchk = GroupMemberChecker(self.p4)
            if gchk.IsMember(self.options.user, prj['allowed_users_groups']):
                self.logger.debug("%s: User allowed to bypass trigger: %s" % (trigger_name, self.options.user))
                return 0

            err_msg = "\n".join(config['msg_cant_create_stream']) +\
                "\nUser: %s" % self.options.user
            self.message(err_msg)
            self.logger.warning(err_msg)
            return 1

        except Exception:
            return self.reportException()

        return 0

if __name__ == '__main__':
    """ Main Program"""
    trigger = ControlStreamCreation(*sys.argv[1:])
    sys.exit(trigger.run())
