#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# ==============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://workshop.perforce.com/view/p4-sdp/main/LICENSE
# ------------------------------------------------------------------------------

# tag::includeManual[]
"""
NAME:
    ExampleTemplate.py

DESCRIPTION:
    This is an example script with test harness for P4 utilities written in python.
    
    It can be used as a base for other scripts.
    
    Test harness is ./test/TestExampleTemplate.py
"""
# end::includeManual[]

# Python 2.7/3.3 compatibility.
from __future__ import print_function

import sys
import argparse
import textwrap
import traceback
import P4
import os
import logging
import traceback
from os.path import basename, splitext

script_name = basename(splitext(__file__)[0])

# If working on a server with the SDP, the 'LOGS' environment variable contains
# the path the standard logging directory.  The '-L <logfile>' argument should
# be specified in non-SDP environments.
LOGDIR = os.getenv('LOGS', '/tmp')

DEFAULT_LOG_FILE = "ExampleTemplate.log"
if os.path.exists(LOGDIR):
    DEFAULT_LOG_FILE = "%s/ExampleTemplate.log" % LOGDIR
DEFAULT_VERBOSITY = 'DEBUG'
LOGGER_NAME = 'ExampleTemplate'


class ExampleTemplate():
    """See module doc string for details"""

    def __init__(self, *args, **kwargs):
        self.p4 = P4.P4(**kwargs)
        self.options = None
        self.parse_args(__doc__, args)

    def parse_args(self, doc, args):
        """Common parsing and setting up of args"""
        desc = textwrap.dedent(doc)
        parser = argparse.ArgumentParser(
            formatter_class=argparse.RawDescriptionHelpFormatter,
            description=desc,
            epilog="Copyright (c) 2022 Perforce Software, Inc."
        )
        self.add_parse_args(parser)     # Should be implemented by subclass
        self.options = parser.parse_args(args=args)
        self.init_logger()
        self.logger.debug("Command Line Options: %s\n" % self.options)

    def add_parse_args(self, parser, default_log_file=None, default_verbosity=None):
        """Default arguments:
        :param default_verbosity:
        :param default_log_file:
        :param parser:
        """
        if not default_log_file:
            default_log_file = DEFAULT_LOG_FILE
        if not default_verbosity:
            default_verbosity = DEFAULT_VERBOSITY
        parser.add_argument('-p', '--port', default=None,
                            help="Perforce server port - set using %%serverport%%. Default: $P4PORT")
        parser.add_argument('-u', '--p4user', default=None, help="Perforce user. Default: $P4USER")
        parser.add_argument('-L', '--log', default=default_log_file, help="Default: " + default_log_file)
        parser.add_argument('-T', '--tickets', help="P4TICKETS file full path")
        parser.add_argument('-v', '--verbosity',
                            nargs='?',
                            const="INFO",
                            default=default_verbosity,
                            choices=('DEBUG', 'WARNING', 'INFO', 'ERROR', 'FATAL'),
                            help="Output verbosity level. Default is: " + default_verbosity)
        # Specific args for this utility (optional)

    def init_logger(self, logger_name=None):
        if not logger_name:
            logger_name = LOGGER_NAME
        self.logger = logging.getLogger(logger_name)
        self.logger.setLevel(self.options.verbosity)
        logformat = '%(levelname)s %(asctime)s %(filename)s %(lineno)d: %(message)s'
        logging.basicConfig(format=logformat, filename=self.options.log, level=self.options.verbosity)

    def message(self, msg):
        """Method to send a message to the user. Just writes to stdout, but it's
        nice to encapsulate that here.
        :param msg: """
        print(msg)

    def reportException(self):
        """Method to encapsulate error reporting to make sure
           all errors are reported in a consistent way"""
        exc_type, exc_value, exc_tb = sys.exc_info()
        self.logger.error("Exception during script execution: %s %s %s" % (exc_type, exc_value, exc_tb))
        self.reportP4Errors()
        self.logger.error("called from:\n%s", "".join(traceback.format_exception(exc_type, exc_value, exc_tb)))
        self.logger.error("port %s user %s tickets %s" % (self.p4.port, self.p4.user, self.p4.ticket_file)) 
        return 1

    def reportP4Errors(self):
        lines = []
        for e in self.p4.errors:
            lines.append("P4 ERROR: %s" % e)
        for w in self.p4.warnings:
            lines.append("P4 WARNING: %s" % w)
        if lines:
            self.message("\n".join(lines))

    def run(self):
        """Runs script"""
        try:
            self.setupP4()
            self.p4.connect()

            # Do whatever the script needs to do
            info = self.p4.run_info()

        except Exception:
            return self.reportException()

        return 0


if __name__ == '__main__':
    """ Main Program"""
    obj = ExampleTemplate(*sys.argv[1:])
    sys.exit(obj.run())
