#!/bin/bash
set -u

# Version ID Block. Relies on +k filetype modifier.
#------------------------------------------------------------------------------
# shellcheck disable=SC2016
declare VersionID='$Id: //p4-sdp/dev_rebrand/test/bsw/test_script-recover_edge_1.sh#2 $ $Change: 31752 $'
declare VersionStream=${VersionID#*//}; VersionStream=${VersionStream#*/}; VersionStream=${VersionStream%%/*};
declare VersionCL=${VersionID##*: }; VersionCL=${VersionCL%% *}
declare Version=${VersionStream}.${VersionCL}
[[ "$VersionStream" == r* ]] || Version="${Version^^}"

declare ThisScript="${0##*/}"
declare -i ErrorCount=0
declare -i Debug=${SDP_DEBUG:-0}
declare EdgeHost=syd-helix-04
declare -i LocalLogCount=0
declare -a LocalLogList
declare -i EdgeLogCount=0
declare -a EdgeLogList
declare EdgeClientsFile=
declare UUID=

function msg () { echo -e "$*"; }
function dbg () { [[ "$Debug" -eq 0 ]] || msg "DEBUG: $*"; }
function errmsg () { msg "\\nError: ${1:-Unknown Error}\\n"; ErrorCount+=1; }
function bail () { errmsg "${1:-Unknown Error}"; exit "$ErrorCount"; }

msg "Start test [$ThisScript version $Version]: Do serial edge dump."

# Setup: Create site config dir if needed
mkdir -p /p4/common/site/config
ssh "$EdgeHost" mkdir -p /p4/common/site/config

msg "Configure for serial checkpoint ops on commit and edge."

{
   echo export DO_PARALLEL_CHECKPOINTS=0
   echo KEEPCKPS=2
   echo KEEPJNLS=2
   echo KEEPCKPS=2
} > /p4/common/site/config/p4_1.vars.local

rsync /p4/common/site/config/p4_1.vars.local "$EdgeHost":/p4/common/site/config/p4_1.vars.local ||\
   errmsg "Failed to copy /p4/common/site/config/p4_1.vars.local."

msg "Create and flush clients on the edge."
UUID=$(uuidgen|cut -c1-6)
p4 -p syd-helix-04:1999 login bruno
p4 -p syd-helix-04:1999 -u bruno --field Root=/p4/1/tmp/Test1.$UUID client -S //jam/dev2.3 -o Test1.$UUID | p4 -s -p syd-helix-04:1999 -u bruno client -i
p4 -p syd-helix-04:1999 -u bruno -c Test1.$UUID flush -q
p4 -p syd-helix-04:1999 -u bruno --field Root=/p4/1/tmp/Test2.$UUID client -S //jam/main -o Test2.$UUID | p4 -s -p syd-helix-04:1999 -u bruno client -i
p4 -p syd-helix-04:1999 -u bruno -c Test2.$UUID flush -q

msg "Rotate journal on commit."
rotate_journal.sh 1
LocalLogList[LocalLogCount]="/p4/1/logs/checkpoint.log"
LocalLogCount+=1

msg "Creating serial edge dump on commit."
msg "Running: edge_dump.sh 1 p4d_edge_syd"
edge_dump.sh 1 p4d_edge_syd

LocalLogList[LocalLogCount]="/p4/1/logs/edge_dump.log"
LocalLogCount+=1

msg "Copying /p4/1/checkpoints to edge server."
rsync -a /p4/1/checkpoints/ syd-helix-04:/p4/1/checkpoints ||\
   errmsg "Could not do: rsync -a /p4/1/checkpoints/ syd-helix-04:/p4/1/checkpoints"

# Get the latest MD5 from the serial dump just created above, which will
# end in *.gz.md5 for a file of the same name sans the .md5 suffix.
EdgeSeedMD5=$(ls -t /p4/1/checkpoints/p4_1.edge_syd.seed.*.gz.md5 | head -1)
EdgeSeed="${EdgeSeedMD5%.md5}"

[[ -f "$EdgeSeed" ]] || errmsg "Could not find checkpoint file: $EdgeSeed"

msg "\\nRunning: ssh $EdgeHost /p4/common/bin/recover_edge.sh 1 ${EdgeSeed}"
ssh "$EdgeHost" /p4/common/bin/recover_edge.sh 1 ${EdgeSeed} ||\
   errmsg "Remote call to recover_edge.sh indicated an error."

EdgeLogList[EdgeLogCount]="/p4/1/logs/recover_edge.log"
EdgeLogCount+=1

msg "\\nChecking that clients created earlier are still here after recovery."
EdgeClientsFile=/p4/1/tmp/edge_clients.txt
p4 -p syd-helix-04:1999 login bruno
p4 -p syd-helix-04:1999 -u bruno clients > "$EdgeClientsFile"
cat "$EdgeClientsFile"
grep Test1.$UUID "$EdgeClientsFile" ||\
   errmsg "Did not see client Test1.$UUID after edge recovery."
grep Test2.$UUID "$EdgeClientsFile" ||\
   errmsg "Did not see client Test2.$UUID after edge recovery."

msg "Checking for syntax and unbound errors in scripts used in this test."
for log in "${LocalLogList[@]}"; do
   msg "Checking for 'unbound variable' in: $log"
   grep ':unbound variable' "$log" && \
      errmsg "Found 'unbound variable' error in $log"
   msg "Checking for 'syntax error' in: $log"
   grep ':syntax error' "$log" && \
      errmsg "Found 'syntax error' in $log"
done

msg "Checking for syntax and unbound errors in scripts used in this test on edge host."
for log in "${EdgeLogList[@]}"; do
   msg "Checking for 'unbound variable' in: $EdgeHost:$log"
   ssh -q "$EdgeHost" "grep ':unbound variable' $log" && \
      errmsg "Found 'unbound variable' error in $log"
   msg "Checking for 'syntax error' in: $EdgeHost:$log"
   ssh -q "$EdgeHost" "grep ':syntax error' $log" && \
      errmsg "Found 'syntax error' in $log"
done

exit "$ErrorCount"
