#!/bin/bash
#------------------------------------------------------------------------------
set -u

# Version ID Block. Relies on +k filetype modifier.
#------------------------------------------------------------------------------
# shellcheck disable=SC2016
declare VersionID='$Id: //p4-sdp/dev_rebrand/test/build_docker_image.sh#2 $ $Change: 31752 $'
declare VersionStream=${VersionID#*//}; VersionStream=${VersionStream#*/}; VersionStream=${VersionStream%%/*};
declare VersionCL=${VersionID##*: }; VersionCL=${VersionCL%% *}
declare Version=${VersionStream}.${VersionCL}
[[ "$VersionStream" == r* ]] || Version="${Version^^}"

declare ThisScript="${0##*/}"
declare ThisUser=
declare ThisHost=${HOSTNAME%%.*}
declare OpMode=Standard
declare BuildCmd=
declare OSList=
declare OS=
declare DockerDir=
declare DockerfileBase=
declare DockerfileSDP=
declare ScriptDir=
declare WorkDir=${HOME}/podman_tmp
declare -i ErrorCount=0

function msg () { echo -e "$*"; }
function errmsg () { msg "\\nError: ${1:-Unknown Error}\\n"; ErrorCount+=1; }
function bail () { errmsg "${1:-Unknown Error}"; exit "$ErrorCount"; }

#------------------------------------------------------------------------------
# Build the Docker containers for the SDP - now using podman instead of docker

# Usage:
#   build_docker_image.sh [<OS>] [-clean]
# Usage Examples from from the root of the workspace (assuming mapping):
#       //guest/perforce_software/sdp/dev/... //YourID.test_sdp_dev.sdp/sdp/...
#
#    sdp/test/build_docker_image.sh
#    sdp/test/build_docker_image.sh ubuntu20
#    sdp/test/build_docker_image.sh centos7 ubuntu20
#    sdp/test/build_docker_image.sh all
#    sdp/test/build_docker_image.sh ALL -clean
#
# Goes together with run_docker_tests.sh
# This is provided as a useful tool for testing!
#
# Note that this file is expecting to be mapped into the root of the workspace
# and with the sdp directory in the same root.
# So workspace view should look something like:
#    View:
#        //guest/perforce_software/sdp/main/... //myws.sdp/sdp/...

# This file should be in <workspace-root>/sdp/test/
# We calculate dir relative to directory of script
ScriptDir="${0%/*}"
# shellcheck disable=SC2164
root_dir="$(cd "$ScriptDir/../.."; pwd -P)"

for arg in $(echo "$@" | tr ',' ' '); do
   case "$arg" in
      (-clean) OpMode=Clean;;
      (all) OSList="rocky9 ubuntu22";;
      (ALL) OSList="centos7 rocky8 rocky9 ubuntu20 ubuntu22";;
      (ubuntu20) OSList+="ubuntu20";;
      (ubuntu22) OSList+="ubuntu22";;
      (centos7) OSList+="centos7";;
      (rocky8) OSList+="rocky8";;
      (rocky9) OSList+="rocky9";;

      (-*)
         echo "Warning: Unknown option [$arg]."
      ;;

      (*)
         echo "Warning: Unknown OS [$arg]."
         OSList+="$arg"
      ;;
   esac
done

# Default is currently Rocky 9 only.
[[ -z "$OSList" ]] && OSList="rocky9"

mkdir -p "$WorkDir" || bail "Could not do: mkdir -p \"$WorkDir\""

export TMPDIR="$WorkDir"

ThisUser=$(id -n -u)
msg "Starting $ThisScript version $Version as $ThisUser@$ThisHost on $(date)."

msg "Building SDP docker containers."
for OS in $OSList; do
    DockerDir="$root_dir/sdp/test/docker"
    Dockerfile="${DockerDir}/Dockerfile.${OS}"
    # Build the base Docker for the OS, and then the SDP variant on top
    BuildCmd="podman build --rm=true -t=perforce/${OS} -f ${Dockerfile} ${DockerDir}"
    [[ "$OpMode" == Clean ]] && BuildCmd=${BuildCmd/--rm=true/--rm=true --no-cache}
    msg "Running: $BuildCmd"
    $BuildCmd || errmsg "Error building SDP container."
done

if [[ "$ErrorCount" -eq 0 ]]; then
   msg "\\nAll processing completed OK."
else
   msg "\\nProcessing completed, but with $ErrorCount errors."
fi

exit "$ErrorCount"
