#!/bin/bash
#==============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://swarm.workshop.perforce.com/projects/perforce-software-sdp/view/main/LICENSE
#------------------------------------------------------------------------------

# This script expects the most recent valid checkpoint to be available in
# $CHECKPOINTS in order for this script to work.
#
# This script is using the following external variables:
#
# SDP_INSTANCE - The instance of Perforce that is being backed up. If not
# set in environment, pass in as argument to script.
#
# P4HOME - Server's home directory.
# P4BIN - Command line client name for the instance being backed up.
# P4DBIN - Server executable name for the instance being backed up.
# P4ROOT - Server's root directory. p4/root, p4_N/root
# P4PORT - TCP/IP port for the server instance being backed up.
# P4JOURNAL - Location of the Journal for the server instance being backed up.
# P4SUPPRESSINFOEMAIL - Suppress information emails. Only send error emails. (Default: false)
#
#
export SDP_INSTANCE=${SDP_INSTANCE:-Undefined}
export SDP_INSTANCE=${1:-$SDP_INSTANCE}
if [[ $SDP_INSTANCE == Undefined ]]; then
   echo "Instance parameter not supplied."
   echo "You must supply the Perforce instance as a parameter to this script."
   exit 1
fi

# If environment variable is defined, then use that as default value if cli arguments not supplied
export P4SUPPRESSINFOEMAIL=${P4SUPPRESSINFOEMAIL:-false}
export P4SUPPRESSINFOEMAIL=${2:-$P4SUPPRESSINFOEMAIL}

# shellcheck disable=SC1091
source /p4/common/bin/p4_vars "$SDP_INSTANCE"
# shellcheck disable=SC1091
source /p4/common/bin/backup_functions.sh
# shellcheck disable=SC2034
LOGFILE="$LOGS/checkpoint.log"

VerifyCmd="$P4CBIN/verify_sdp.sh $SDP_INSTANCE -skip ${VERIFY_SDP_SKIP_TEST_LIST:-},crontab,excess,init,license,masterid,version -L off -online"

######### Start of Script ##########

check_vars
set_vars
rotate_last_run_logs
log "Start $P4SERVER Checkpoint"

log "Calling 'verify_sdp.sh' before running daily checkpoint:\\n$VerifyCmd"
$VerifyCmd >> "$LOGFILE" 2>&1 ||\
   die "Error: Cannot confirm all is well with $P4CBIN/verify_sdp.sh. Aborting."

check_uid
check_dirs
check_offline_db_usable
ckp_running
"$P4CBIN"/p4login
get_journalnum
get_offline_journal_num

if [[ "$SERVERID" == "$P4MASTER_ID" ]]; then
   truncate_journal
else
   log "Skipping call to truncate_journal() on edge or replica server."
fi

replay_journals_to_offline_db
# shellcheck disable=SC2034
ROOTDIR="$OFFLINE_DB"
dump_checkpoint
# SDP-761: Tech Preview feature.
if [[ "${SDP_DAILY_OFFLINE_DB_REGEN:-1}" == "1" ]]; then
   recreate_offline_db_files
else
   log "Skipping regeneration of db.* files in offline_db."
fi
remove_old_checkpoints_and_journals
check_disk_space
remove_old_logs
set_counter
log "End $P4SERVER Checkpoint"
if [[ "${P4SUPPRESSINFOEMAIL,,}" == "false" ]]; then
   mail_log_file "$HOSTNAME $P4SERVER Daily maintenance log."
fi
ckp_complete

