#!/bin/bash
#==============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://swarm.workshop.perforce.com/projects/perforce-software-sdp/view/main/LICENSE
#------------------------------------------------------------------------------

# This script is designed to create a seed checkpoint for an Edge server.
#
# An edge server is naturally filtered, with certain database tables (e.g.
# db.have) excluded.  In addition to implicit filtering, the server spec may
# specify additional tables to be excluded, e.g. by using the
# ArchiveDataFilter field of the server spec.
#
# The script requires the SDP instance and the edge ServerID.

# Run example:
#  ./edge_dump.sh 1 p4d_edge_syd

set -u

declare EdgeSeedCheckpoint=
declare ExcludedTables=
declare DumpCmd=
declare LogLink=
declare -i Threads=4

function usage () {
   echo -e "Usage:\n\t${0##*/} <SDP_Instance> <ServerID>\n"
   exit 1
}

[[ $# -ne 2 || ${1:-Unset} == -h ]] && usage

export SDP_INSTANCE=${SDP_INSTANCE:-Undefined}
export SDP_INSTANCE=${1:-$SDP_INSTANCE}
if [[ $SDP_INSTANCE == Undefined ]]; then
   echo -e "Usage Error: Instance parameter not supplied."
   usage
fi

declare ServerID=${2:-Unset}
if [[ $ServerID == Unset ]]; then
   echo -e "Usage Error: ServerID parameter not supplied."
   usage
fi

#shellcheck disable=SC1091
source /p4/common/bin/p4_vars "$SDP_INSTANCE"
#shellcheck disable=SC1091
source /p4/common/bin/backup_functions.sh
#shellcheck disable=SC1091
source /p4/common/bin/edge_vars
export LOGFILE
# shellcheck disable=SC1091
source /p4/common/bin/log_functions.sh

export LOGFILE
LOGFILE="$LOGS/edge_dump.$(date +'%Y-%m-%d-%H%M%S').log"

# The LogLink symlink has no timestamp. It points to the most recent log file.
LogLink="$LOGS/edge_dump.log"

######### Start of Script ##########

if [[ -e "$LogLink" ]]; then
   if [[ -L "$LogLink" ]]; then
      rm -f "$LogLink"
   else
      # If the name that should be a symlink is not a symlink, move it aside before
      # creating the symlink.
      OldLogTimestamp=$(get_old_log_timestamp "$LogLink")
      mv -f "$LogLink" "${LogLink%.log}.${OldLogTimestamp}.log"
   fi
fi

# Point LogLink symlink to current log. Use a subshell so the 'cd' doesn't persist.
( cd "$LOGS" && ln -s "${LOGFILE##*/}" "${LogLink##*/}"; )

echo "Processing. This may take a while depending on checkpoint duration."
echo "Log file is: $LOGFILE"

check_vars
set_vars
check_offline_db_usable
get_offline_journal_num

if [[ "$(is_edge "$ServerID" "$OFFLINE_DB")" == YES ]]; then
   log "Verified: Server $ServerID is an edge server."
else
   die "Could not confirm that $ServerID is an edge server."
fi

[[ -n "$ExcludedTables" ]] ||\
   die "Value for \$ExcludedTables not defined in $P4CBIN/edge_vars."

EdgeSeedCheckpoint="${CHECKPOINTS}/${P4SERVER}.${ServerID#p4d_}.seed.$OFFLINEJNLNUM"

if [[ -f "${EdgeSeedCheckpoint}" ]]; then
   die "${EdgeSeedCheckpoint} already exists."
fi

ckp_running

# shellcheck disable=SC2072
if [[ -n "${DO_PARALLEL_CHECKPOINTS:-}" && "$DO_PARALLEL_CHECKPOINTS" != "0" && "$P4D_VERSION" > "2022.2" ]]; then
   if [[ "$DO_PARALLEL_CHECKPOINTS" =~ ^[1-9]{1}[0-9]*$ ]]; then
      Threads="$DO_PARALLEL_CHECKPOINTS"
   else
      Threads=4
   fi
   DumpCmd="$P4DBIN -r $OFFLINE_DB -K $ExcludedTables -P $ServerID -N $Threads -jdp -z $EdgeSeedCheckpoint"
else
   DumpCmd="$P4DBIN -r $OFFLINE_DB -K $ExcludedTables -P $ServerID -jd -z ${EdgeSeedCheckpoint}.gz"
fi

log "Create edge seed checkpoint from master offline_db skipping edge-local tables and using the edge server form for any filtered data."
log "Running: $DumpCmd"
{ time $DumpCmd ; } >> "$LOGFILE" 2>&1 || { die "Edge seed checkpoint dump failed!"; }

log "Edge seed checkpoint complete."

# Inform the user about follow up tasks
log "You now need to copy $EdgeSeedCheckpoint to the edge server and use /p4/common/bin/recover_edge.sh to reseed the edge server."

ckp_complete

cat "$LOGFILE"
