#!/bin/bash
#==============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://swarm.workshop.perforce.com/projects/perforce-software-sdp/view/main/LICENSE
#------------------------------------------------------------------------------

# This script is for use only on a standby replica.  It will not run on any
# other type of replica.

# This script ensures the offline_db has the most current journals replayed.

# It is intended for use on standby replicas as an alternative to sync_replica.sh
# or replica_cleanup.sh.  It is ideal for use in an environment where the
# checkpoints folder of the P4TARGET server is shared (e.g. via NFS) with this
# server.

# This script does NOT do full checkpoint operations, and requires that the
# offline_db be in a good state before starting -- this is verified with
# a call to verify_sdp.sh.

# This uses checkpoint.log as its primary log. It is only intended for use
# on a machine where other scripts that update checkpoint.log don't run, e.g.
# daily_checkpoint.sh, live_checkpoint.sh, or rotate_journal.sh.

export SDP_INSTANCE=${SDP_INSTANCE:-Undefined}
export SDP_INSTANCE=${1:-$SDP_INSTANCE}
if [[ $SDP_INSTANCE == Undefined ]]; then
   echo "Instance parameter not supplied."
   echo "You must supply the Perforce instance as a parameter to this script."
   exit 1
fi

# If environment variable is defined, then use that as default value if cli arguments not supplied
export P4SUPPRESSINFOEMAIL=${P4SUPPRESSINFOEMAIL:-false}
export P4SUPPRESSINFOEMAIL=${2:-$P4SUPPRESSINFOEMAIL}

# shellcheck disable=SC1091
source /p4/common/bin/p4_vars "$SDP_INSTANCE"
# shellcheck disable=SC1091
source /p4/common/bin/backup_functions.sh
# shellcheck disable=SC2034
LOGFILE="$LOGS/checkpoint.log"

VerifyCmd="$P4CBIN/verify_sdp.sh $SDP_INSTANCE -skip ${VERIFY_SDP_SKIP_TEST_LIST:-},crontab,excess,init,license,masterid,version -L off"

######### Start of Script ##########

check_vars
set_vars

if [[ "$(is_standby "$SERVERID")" != YES ]]; then
   die "Error: Run this only on a standby replica."
fi

rotate_last_run_logs
log "Start $P4SERVER Keep offline_db current"

log "Calling 'verify_sdp.sh' before replaying journal to offline_db:\\n$VerifyCmd"
$VerifyCmd >> "$LOGFILE" 2>&1 ||\
   die "Error: Cannot confirm all is well with $P4CBIN/verify_sdp.sh. Aborting."

check_uid
check_dirs
check_offline_db_usable
ckp_running
get_journalnum
get_offline_journal_num

# Replay journals based on the P4TARGET server's journal prefix, which may
# or may not be NFS mounted, but is presumed to be current.
replay_journals_to_offline_db 1

# Remove old logs and journals based on the standby server's own journalPrefix,
# which will remove excess journals p4d has been rotating.
remove_old_checkpoints_and_journals
check_disk_space
remove_old_logs
log "End $P4SERVER Keep offline_db current"
if [[ "${P4SUPPRESSINFOEMAIL,,}" == "false" ]]; then
   mail_log_file "$HOSTNAME $P4SERVER Daily maintenance log."
fi
ckp_complete
