# ============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://swarm.workshop.perforce.com/projects/perforce-software-sdp/view/main/LICENSE
# ----------------------------------------------------------------------------

# tag::includeManual[]
<#
    .Synopsis
        p4verify.ps1 performs p4d archive verification on all appropriate depots
        
    .Description
        Runs "p4 verify -qz //depot/..." as appropriate to the depot type.
        For replicas it adds the "-t" flag to transfer missing revisions.
        Sends an error message with errors if BAD! or MISSING! is found in the output.

    .Parameter sdp-instance
        The specified SDP instance to verify

    .Option -fast
        Specify -fast to check only for MISSING files, skipping expensive checksum
        calculations required for a full verify. This requires the P4D server
        version to be 2021.1 or later.
        
    .Option -nt
        Specify -nt to avoid scheduling tranfers on a replica. By default, if running
        on a replica, the '-t' option is added to 'p4 verify' commands to schedule
        transfers to attempt self-healing missing or bad archive files.  The -nt can
        be used to confirm whether prior attempts at self-healing by transferring
        files from the P4TARGET server were effective.
        
    .Example
        p4verify.ps1 1

    .Example
        p4verify.ps1 1 -fast

    .Example
        p4verify.ps1 1 -nt
#>
# end::includeManual[]

[CmdletBinding()]
param (
    [string]$SDPInstance = $(throw "SDPInstance parameter is required."),
    [switch]$fast = $false,
    [switch]$nt = $false
)

Set-StrictMode -Version 2.0

# Source the SDP Functions shared between scripts
$SDPFunctionsPath = Split-Path -parent $MyInvocation.MyCommand.Path | Join-Path -childpath "SDP-Functions.ps1" 
. $SDPFunctionsPath

$global:ScriptName = "P4Verify.ps1"
$global:ScriptTask = "Verify"
$global:LogFileName = "p4verify.log"

Parse-SDPConfigFile $MyInvocation.MyCommand.Path
Create-LogFile

$OrigPath = convert-path .
Set-Location -Path $global:LOGS_DIR

Function Run-Verify () {
    Log "Running verify"
    $depots = & $global:P4exe -p $global:P4PORT -u $global:SDP_P4SUPERUSER depots 2>&1
    Log $depots
    if ($lastexitcode -ne 0) {
        throw "ERROR - failed to get depots!"
    }
    # If a replica then add -t for transfer to args, unless '-nt' was specified.
    $replica_args = ""
    if (is-replica) {
        if ( -Not $nt) {
            $replica_args = "-t"
        }
    }
    $fast_args = ""
    if ($fast) {
        $fast_args = "--only MISSING"
    }
    foreach ($dline in $depots) {
        $dname = $dline.split()[1]
        $dtype = $dline.split()[3]
        Log "$dname has type $dtype"
        if (!($dtype -match "remote|archive")) {
            if ($dtype -match "unload") {
                $cmd = "$global:P4exe -p $global:P4PORT -s -u $global:SDP_P4SUPERUSER verify -U -q $replica_args $fast_args //$dname/... >> $global:LogFile 2>&1"
                Log $cmd
                cmd /c $cmd
            } else {
                if ($fast) {
                   # Don't use '-z' if we're only verifying MISSING file with '--only MISSING'.
                   $cmd = "$global:P4exe -p $global:P4PORT -s -u $global:SDP_P4SUPERUSER verify -q $replica_args $fast_args //$dname/... >> $global:LogFile 2>&1"
                } else {
                   $cmd = "$global:P4exe -p $global:P4PORT -s -u $global:SDP_P4SUPERUSER verify -qz $replica_args //$dname/... >> $global:LogFile 2>&1"
                }
                Log $cmd
                cmd /c $cmd
                $cmd = "$global:P4exe -p $global:P4PORT -s -u $global:SDP_P4SUPERUSER verify -qS $replica_args $fast_args //$dname/... >> $global:LogFile 2>&1"
                Log $cmd
                cmd /c $cmd
            }
        }
    }
    # Finally we check for errors
    $matches = Get-Content $global:logfile | Select-String -Pattern "BAD!|MISSING!" -AllMatches
    if ($matches -ne $null) {
        $count = $matches.matches.count
        if ($count -gt 0) {
            throw "Found $count verify errors in logfile - contact support-helix-core@perforce.com if you need help to resolve"
        }
    }
}

try {
    Log "If there are errors in this log, you may wish to contact support-helix-core@perforce.com for help."
    Invoke-P4Login
    Run-Verify
    Remove-OldLogs
    Log "End ${global:SDP_INSTANCE_P4SERVICE_NAME} ${global:ScriptTask}"
    Send-Email "${env:computername} ${global:SDP_INSTANCE_P4SERVICE_NAME} ${global:ScriptTask} log."
    Write-Output "`r`n${global:ScriptTask} completed successfully - see ${global:logfile}"
}
Catch
{
    write-error $error[0].ScriptStackTrace
    LogException $_.Exception
    Send-Email "FAILED: ${env:computername} ${global:SDP_INSTANCE_P4SERVICE_NAME} ${global:ScriptTask} log."
    Write-Output "`r`nFAILED - ${global:ScriptTask} - see ${global:logfile}"
}

Set-Location -Path $OrigPath
