# ============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://swarm.workshop.perforce.com/projects/perforce-software-sdp/view/main/LICENSE
# ----------------------------------------------------------------------------

# tag::includeManual[]
<#
    .Synopsis
        Create-offline-db-from-checkpoint.ps1 recreates offline_db using latest checkpoint found.
        
    .Description
        This script recreates offline_db files from the latest checkpoint. If it fails, then
        check to see if the most recent checkpoint in the c:\p4\<INSTANCE>\checkpoints directory is
        bad (ie doesn't look like the right size compared to the others), and if so, delete it
        and rerun this script. If the error you are getting is that the journal replay failed,
        then the only option is to run live-checkpoint.ps1 script (which locks database while
        it runs).
        
    .Parameter sdp-instance
        The specified instance to process
        
    .Example
        Create-offline-db-from-checkpoint.ps1 Master
        
    .Example
        Create-offline-db-from-checkpoint.ps1 1
#>
# end::includeManual[]

[CmdletBinding()]
param ([string]$SDPInstance = $(throw "SDPInstance parameter is required."))

Set-StrictMode -Version 2.0

# Source the SDP Functions shared between scripts
$SDPFunctionsPath = Split-Path -parent $MyInvocation.MyCommand.Path | Join-Path -childpath "SDP-Functions.ps1" 
. $SDPFunctionsPath

$global:ScriptName = "Create-offline-db-from-checkpoint.ps1"
$global:ScriptTask = "Recreate offline db from checkpoint"
$global:LogFileName = "recreate-offline-db-from-checkpoint.log"

Parse-SDPConfigFile $MyInvocation.MyCommand.Path
Create-LogFile

$OrigPath = convert-path .
Set-Location -Path $global:LOGS_DIR

try {
    Ensure-CheckpointNotRunning
    Get-CurrentJournalCounter
    # Drop the journal number by one since we are not truncating the journal and
    # replay_journals_to_offline_db assumes a truncate_journal has been run.
    $global:JOURNAL_NUM = $global:JOURNAL_NUM - 1
    Recreate-OfflineDBFiles
    Invoke-P4Login
    Get-OfflineJournalCounter
    Replay-JournalsToOfflineDB
    # Don't remove other files such as logs etc
    Log "End ${global:SDP_INSTANCE_P4SERVICE_NAME} ${global:ScriptTask}"
    Signal-CheckpointComplete
    Write-Output "`r`n${global:ScriptTask} completed successfully - see ${global:logfile}"
}
Catch
{
    write-error $error[0].ScriptStackTrace
    LogException $_.Exception
    Write-Output "`r`nFAILED - ${global:ScriptTask} - see ${global:logfile}"
}

Set-Location -Path $OrigPath
