# ============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://swarm.workshop.perforce.com/projects/perforce-software-sdp/view/main/LICENSE
# ----------------------------------------------------------------------------

# tag::includeManual[]
<#
    .Synopsis
        Rotate-journal.ps1 performs journal rotation on live and then applies it to offline database.
        
    .Description
        Admin access is required.
        This is the first part of daily-backup.ps1 (so it doesn't create checkpoint from offline_db)
        This can be scheduled on sites where a fully daily-backup places too much load on the server
        (and takes too long so runs during working day)
        
    .Parameter sdp-instance
        The specified instance to backup
        
    .Example
        rotate-journal.ps1 Master
        
    .Example
        rotate-journal.ps1 1
#>
# end::includeManual[]

[CmdletBinding()]
param ([string]$SDPInstance = $(throw "SDPInstance parameter is required."))

Set-StrictMode -Version 2.0

# Source the SDP Functions shared between scripts
$SDPFunctionsPath = Split-Path -parent $MyInvocation.MyCommand.Path | Join-Path -childpath "SDP-Functions.ps1" 
. $SDPFunctionsPath

$global:ScriptName = "Rotate-journal.ps1"
$global:ScriptTask = "Rotate Journal"
$global:LogFileName = "rotate_journal.log"

Parse-SDPConfigFile $MyInvocation.MyCommand.Path
Create-LogFile

$OrigPath = convert-path .
Set-Location -Path $global:LOGS_DIR

try {
    Invoke-P4Login
    Check-OfflineDBUsable
    Check-OfflineDBExists
    Ensure-CheckpointNotRunning
    Get-CurrentJournalCounter
    Get-OfflineJournalCounter
    Truncate-Journal
    Rotate-Logfiles
    Replay-JournalsToOfflineDB
    Remove-OldCheckpointsAndJournals
    Log-DiskSpace
    Remove-OldLogs
    Log "End ${global:SDP_INSTANCE_P4SERVICE_NAME} ${global:ScriptTask}"
    Send-Email "${env:computername} ${global:SDP_INSTANCE_P4SERVICE_NAME} ${global:ScriptTask} log."
    Set-Counter
    Signal-CheckpointComplete
    Write-Output "`r`n${global:ScriptTask} completed successfully - see ${global:logfile}"
}
Catch
{
    write-error $error[0].ScriptStackTrace
    LogException $_.Exception
    Send-Email "FAILED: ${env:computername} ${global:SDP_INSTANCE_P4SERVICE_NAME} ${global:ScriptTask} log."
    Write-Output "`r`nFAILED - ${global:ScriptTask} - see ${global:logfile}"
}

Set-Location -Path $OrigPath
