#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# ==============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://swarm.workshop.perforce.com/projects/perforce-software-sdp/view/main/LICENSE
# ------------------------------------------------------------------------------

# tag::includeManual[]
"""
NAME:
    CheckSubmitHasReview.py

DESCRIPTION:
    This trigger is intended for use with Swarm installations.

    It will make sure that any changelist being submitted has an associated Swarm review.
    Note it is possible to configure a list of users who are allowed to bypass the trigger,
    e.g. a jenkins user who submits after building. See Workflow.yaml

    To install, add a line to your Perforce triggers table like the following:

        create_swarm_review change-submit //... "python /p4/common/bin/triggers/CheckSubmitHasReview.py -c /p4/common/config/Workflow.yaml -p %serverport% -u perforce %change% "

    or (if server is standard SDP and has appropriate environment defaults for P4PORT and P4USER):

        create_swarm_review change-submit //... "python /p4/common/bin/triggers/CheckSubmitHasReview.py -c /p4/common/config/Workflow.yaml %change% "

    Note that -c/--config specifies a yaml file - see Workflow.yaml for example.

    You may need to provide the full path to python executable, or edit the path to the trigger.

    Also, don't forget to make the file executable.
"""
# end::includeManual[]

# Python 2.7/3.3 compatibility.
from __future__ import print_function

import sys

import WorkflowTriggers
import P4
import requests
import os

trigger_name = os.path.basename(os.path.splitext(__file__)[0])

class CheckSubmitHasReview(WorkflowTriggers.WorkflowTrigger):
    """See module doc string for details"""

    def __init__(self, *args, **kwargs):
        WorkflowTriggers.WorkflowTrigger.__init__(self, **kwargs)
        self.parse_args(__doc__, args)

    def add_parse_args(self, parser):
        """Specific args for this trigger - also calls super class to add common trigger args"""
        parser.add_argument('-c', '--config-file', default=None,
                            help="Configuration file for trigger. Default: CheckSubmitHasReview.yaml")
        parser.add_argument('--test-mode', default=False, action='store_true',
                            help="Trigger runs in test mode - for unit testing only")
        parser.add_argument('change', help="Change to process - %%change%% argument from triggers entry.")
        super(CheckSubmitHasReview, self).add_parse_args(parser)

    def run(self):
        """Runs trigger"""
        try:
            self.logger.debug("%s: trigger firing" % trigger_name)

            config = self.load_config(self.options.config_file)
            errors = []
            for k in "msg_submit_requires_review msg_submit_test_must_pass".split():
                if k not in config:
                    errors.append("Config file %s missing definition for %s" %
                                  (self.options.config_file, k))
            if errors:
                msg = "Invalid config file for trigger" + str(errors) + "\n"
                self.message(msg)
                return 1

            self.setupP4()
            self.p4.connect()
            change = self.getChange(self.options.change)

            if 'submit_without_review_users' in config:
                for user in config['submit_without_review_users']:
                    if user == change.user:
                        self.logger.debug("%s: User allowed to submit without review: %s" % (trigger_name, user))
                        return 0

            swarm_url = self.get_swarm_url()
            base_url = "%s%s" % (swarm_url, config['api'])
            auth = (config['user'], config['ticket'])

            # Now get any reviews for the specified change
            if self.project_flag_true(config, change, 'pre_submit_require_review'):
                testRequired = self.project_flag_true(config, change, 'pre_submit_test_must_pass')
                chg_search = "change[]=%s" % self.options.change
                url = '%s/reviews?%s' % (base_url, chg_search)
                self.logger.debug("Get: %s" % (url))
                resp = requests.get(url, auth=auth)
                resp_json = resp.json()
                self.logger.debug("Result: %s" % (resp_json))
                if 'reviews' in resp_json:
                    reviews = [str(r['id']) for r in resp_json['reviews']]
                    self.logger.debug("Reviews: %s" % (reviews))
                    if len(reviews) > 1:
                        self.logger.warning("Multiple reviews found")
                    if reviews and len(reviews) > 0:
                        if testRequired:
                            # Make sure testStatus is 'pass'
                            for r in resp_json['reviews']:
                                if not 'testStatus' in r or not r['testStatus'] or \
                                    r['testStatus'] != 'pass':
                                    err_msg = "\n".join(config['msg_submit_test_must_pass'])
                                    err_msg += "\n%sreviews/%s" % (swarm_url, r['id']) 
                                    self.message(err_msg)
                                    return 1
                            self.logger.debug("Valid testStatus found for review")
                        self.logger.debug("Valid submit as 1 or more reviews found")
                        return 0
                err_msg = "\n".join(config['msg_submit_requires_review'])
                self.message(err_msg)
                return 1

        except Exception:
            return self.reportException()

        return 0


if __name__ == '__main__':
    """ Main Program"""
    trigger = CheckSubmitHasReview(*sys.argv[1:])
    sys.exit(trigger.run())
