#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# ==============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://swarm.workshop.perforce.com/projects/perforce-software-sdp/view/main/LICENSE
# ------------------------------------------------------------------------------

# tag::includeManual[]
"""
NAME:
    DefaultChangeDesc.py

DESCRIPTION:
    Create a default change description (form-out)

    Can be used together with CheckChangeDesc.py (change-submit)

    To install, add a line to your Perforce triggers table like the following:

        default-change-desc form-out change "python /p4/common/bin/triggers/DefaultChangeDesc.py -p %serverport% -u perforce %user% %formfile% "

    or (if server is standard SDP and has appropriate environment defaults for P4PORT and P4USER):

        default-change-desc form-out change "python /p4/common/bin/triggers/DefaultChangeDesc.py %user% %formfile% "

    You may need to provide the full path to python executable, or edit the path to the trigger.

    Also, don't forget to make the file executable, and set the configuration variables below.

    To configure, read and modify the following lines up to the comment that
    reads "END OF CONFIGURATION BLOCK".  You may also need to modify the
    definition of which fields constitute a new job based on your jobspec.  This
    is in the allowed_job() function.
"""
# end::includeManual[]

# Python 2.7/3.3 compatibility.
from __future__ import print_function

import sys
import re
import argparse
import textwrap
import P4Triggers
import shutil
import os

######################################################################
# tag::includeManual[]
# CONFIGURATION BLOCK

# The error messages we give to the user
DEFAULT_CHANGE_DESC = """OS-NNNN/WF-NNNN <Description of change>

# Please choose either OS or WF JIRA issue ID and your description
# Description may be multi line.
# Comment lines like these beginning with '#' can be edited out or will
# be removed automatically when change is submitted.
# The description will be validated before you save.
"""

# END OF CONFIGURATION BLOCK
# end::includeManual[]
######################################################################

class DefaultChangeDesc(P4Triggers.P4Trigger):
    """See module doc string for details"""

    def __init__(self, *args, **kwargs):
        P4Triggers.P4Trigger.__init__(self, **kwargs)
        self.parse_args(__doc__, args)

    def add_parse_args(self, parser):
        """Specific args for this trigger - also calls super class to add common trigger args"""
        parser.add_argument('user', help="User carrying out the command - %%user%% argument from triggers entry.")
        parser.add_argument('formfile',
                            help="Formfile containing job definition - %%formfile%% argument from triggers entry.")
        super(DefaultChangeDesc, self).add_parse_args(parser)


    def run(self):
        """Runs trigger"""
        try:
            self.logger.debug("DefaultChangeDesc firing")
            self.setupP4()
            self.p4.connect()
            with open(self.options.formfile, 'r') as f:
                contents = f.read()
            chg = self.p4.parse_change(contents)
            if not chg["Description"].startswith("<enter description here>"):
                return 0
            chg["Description"] = DEFAULT_CHANGE_DESC
            contents = self.p4.format_change(chg)
            tempfile = "%s~" % self.options.formfile
            shutil.move(self.options.formfile, tempfile)
            with open(self.options.formfile, 'w') as f:
                f.write(contents)
            os.remove(tempfile)
        except Exception:
            return self.reportException()

        return 0


if __name__ == '__main__':
    """ Main Program"""
    trigger = DefaultChangeDesc(*sys.argv[1:])
    sys.exit(trigger.run())