#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# ==============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://swarm.workshop.perforce.com/projects/perforce-software-sdp/view/main/LICENSE
# ------------------------------------------------------------------------------

# tag::includeManual[]
"""
NAME:
    CheckChangeDesc.py

DESCRIPTION:
    Check the format of a change description on submit (change-submit)

    Can be used together with DefaultChangeDesc.py (form-out)

    To install, add a line to your Perforce triggers table like the following:

        check-change-desc change-submit //... "python /p4/common/bin/triggers/CheckChangeDesc.py -p %serverport% -u perforce %change% "

    or (if server is standard SDP and has appropriate environment defaults for P4PORT and P4USER):

        check-change-desc change-submit //... "python /p4/common/bin/triggers/CheckChangeDesc.py %change% "

    You may need to provide the full path to python executable, or edit the path to the trigger.

    Also, don't forget to make the file executable, and set the configuration variables below.

    To configure, read and modify the following lines up to the comment that
    reads "END OF CONFIGURATION BLOCK".  You may also need to modify the
    definition of which fields constitute a new job based on your jobspec.  This
    is in the allowed_job() function.
"""
# end::includeManual[]

# Python 2.7/3.3 compatibility.
from __future__ import print_function

import sys
import re
import argparse
import textwrap
import P4Triggers
import shutil
import os

######################################################################
# CONFIGURATION BLOCK

# tag::includeManual[]
# The error messages we give to the user
ERROR_MSG = """
First line of changelist description must be:

          OS-NNNN and some text
or
          WF-NNNN and some text

Other lines of text allowed.
"""
# end::includeManual[]

# END OF CONFIGURATION BLOCK
######################################################################

class CheckChangeDesc(P4Triggers.P4Trigger):
    """See module doc string for details"""

    def __init__(self, *args, **kwargs):
        P4Triggers.P4Trigger.__init__(self, **kwargs)
        self.parse_args(__doc__, args)

    def add_parse_args(self, parser):
        """Specific args for this trigger - also calls super class to add common trigger args"""
        parser.add_argument('change', help="Change to validate - %%change%% argument from triggers entry.")
        super(CheckChangeDesc, self).add_parse_args(parser)

    def run(self):
        """Runs trigger"""
        try:
            self.logger.debug("CheckChangeDesc firing")
            self.setupP4()
            self.p4.connect()
            change = self.getChange(self.options.change, "-f")
            valid_pattern = re.compile("^OS\-[0-9]+\s+[\S+].*|^WF\-[0-9]+\s+[\S+].*")
            if not valid_pattern.search(change.desc):
                print(ERROR_MSG)
                return(1)
        except Exception:
            return self.reportException()

        return 0


if __name__ == '__main__':
    """ Main Program"""
    trigger = CheckChangeDesc(*sys.argv[1:])
    sys.exit(trigger.run())