#!/bin/bash
set -u

# Version ID Block. Relies on +k filetype modifier.
#------------------------------------------------------------------------------
# shellcheck disable=SC2016
declare VersionID='$Id: //p4-sdp/dev_c2s/Server/Unix/p4/common/bin/check_dir_ownership.sh#2 $ $Change: 31472 $'
declare VersionStream=${VersionID#*//}; VersionStream=${VersionStream#*/}; VersionStream=${VersionStream%%/*};
declare VersionCL=${VersionID##*: }; VersionCL=${VersionCL%% *}
declare Version=${VersionStream}.${VersionCL}
[[ "$VersionStream" == r* ]] || Version="${Version^^}"

declare ThisScript="${0##*/}"
declare ThisUser=
declare ThisHost=${HOSTNAME%%.*}
declare -i ErrorCount=0

#==============================================================================
# Local Functions

# Note: This script does not use SDP library files, as its purpose is to
# upgrade the SDP installation.
function msg () { echo -e "$*"; }
function errmsg () { msg "\\nError: ${1:-Unknown Error}\\n"; ErrorCount+=1; }
###function bail () { errmsg "${1:-Unknown Error}"; exit "${2:-1}"; }

#------------------------------------------------------------------------------
# Function: check_ownership ($dir, $expectedUser, $expectedGroup)
# $dir - directory to check.
# $expectedUser - Expected user.
# $expectedGroup - Expected group ownership. Optional; if not specified,
# group check is skipped.
check_ownership() {
   local dir="${1:-}"
   local expectedUser="${2:-}"
   local expectedGroup="${3:-unset}"
   local maxDepth="${4:-}"

   # Check if required arguments are provided
   if [[ -z "$dir" || -z "$expectedUser" || -z "$expectedGroup" ]]; then
      msg "Usage: check_ownership <directory> <expectedUser> [<expectedGroup>|unset] [<maxDepth>]"
      return 2
   fi

   # Ensure the directory path ends with a '/'
   [[ "${dir: -1}" != "/" ]] && dir="${dir}/"

   if [[ -n "$maxDepth" ]]; then
      msg "Checking ownership of directory: $dir (max depth of $maxDepth)"
   else
      msg "Checking ownership of directory: $dir"
   fi
   msg "Expected user: $expectedUser, Expected group: $expectedGroup"

   # Traverse directory, handling inaccessible files gracefully
   if [[ "$expectedGroup" != unset ]]; then
      if [[ -n "$maxDepth" ]]; then
         anomalies=$(find "$dir" -maxdepth $maxDepth -exec stat --format="%U %G %n" {} + 2>/dev/null | awk -v user="$expectedUser" -v group="$expectedGroup" '
         $1 != user || $2 != group {
            print $1, $2, $3
         }')
      else
         anomalies=$(find "$dir" -exec stat --format="%U %G %n" {} + 2>/dev/null | awk -v user="$expectedUser" -v group="$expectedGroup" '
         $1 != user || $2 != group {
            print $1, $2, $3
         }')
      fi
   else
      if [[ -n "$maxDepth" ]]; then
         anomalies=$(find "$dir" -maxdepth "$maxDepth" -exec stat --format="%U %n" {} + 2>/dev/null | awk -v user="$expectedUser" '
         $1 != user {
            print $1, $2
         }')
      else
         anomalies=$(find "$dir" -exec stat --format="%U %n" {} + 2>/dev/null | awk -v user="$expectedUser" '
         $1 != user {
            print $1, $2
         }')
      fi
   fi

   # Check if anomalies were found.
   if [[ -z "$anomalies" ]]; then
      msg "No ownership anomalies found."
      return 0
   fi

   # Display anomalies and suggest corrections
   msg "Found ownership anomalies:"
      if [[ "$expectedGroup" != unset ]]; then
      echo "$anomalies" | while read -r actualUser actualGroup file; do
         errmsg "$file is owned by $actualUser:$actualGroup, expected $expectedUser:$expectedGroup."
         if [[ -L "$file" ]]; then
            # Command to change ownership of symlink
            msg "Suggested fix: sudo chown -h $expectedUser:$expectedGroup $file"
         else
            # Command to change ownership of regular file or directory
            msg "Suggested fix: sudo chown $expectedUser:$expectedGroup $file"
         fi
      done
   else
      echo "$anomalies" | while read -r actualUser file; do
         errmsg "$file is owned by $actualUser, expected $expectedUser."
         if [[ -L "$file" ]]; then
            # Command to change ownership of symlink
            msg "Suggested fix: sudo chown -h $expectedUser $file"
         else
            # Command to change ownership of regular file or directory
            msg "Suggested fix: sudo chown $expectedUser $file"
         fi
      done
   fi

   return 1
}

# Call the function if executed directly.
if [[ "${BASH_SOURCE[0]}" == "$0" ]]; then
   ThisUser=$(id -n -u)
   msg "Starting $ThisScript version $Version as $ThisUser@$ThisHost at $(date +'%a %Y-%m-%d %H:%M:%S %Z')"
   check_ownership "$@" || exit 1
   exit 0
fi
