#!/bin/bash
#==============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://swarm.workshop.perforce.com/projects/perforce-software-sdp/view/main/LICENSE
#------------------------------------------------------------------------------

#------------------------------------------------------------------------------
# Version ID Block. Relies on +k filetype modifier.
# VersionID='$Id: //p4-sdp/dev_c2s/Unsupported/Samples/bin/cache_purge.sh#2 $ $Change: 31472 $'

# This script will allow you to run the p4 cachepurge command via a crontab.
# https://www.perforce.com/manuals/cmdref/Content/CmdRef/p4_cachepurge.html
#
# The idea is to specify the amount of free space you want to maintain in the
# depot filesystem and then log activity.
#
# Parameters
#
# SDP_INSTANCE - The instance of Perforce that is being backed up. If not
# set in environment, pass in as argument to script.
#
# FREE_PERCENTAGE - Integer value
#
# USAGE: ./cache_purge.sh -i <SDP_INSTANCE> -f <PERCENTAGE_FREE>
#
# EXAMPLES:
#   Run from CLI that will archive files as defined in the config file
#      ./cache_purge.sh -i 1 -f 10
#
#   Cron job that will will archive files as defined in the config file, including text files
#      30 0 * * * /p4/common/bin/run_if_edge.sh ${INSTANCE} /p4/common/site/bin/cache_purge.sh -i ${INSTANCE} -f 10
#

function msg () { echo -e "$*"; }
function bail () { msg "\nError: ${1:-Unknown Error}\n"; exit ${2:-1}; }

function usage
{
   declare style=${1:--h}
   declare errorMessage=${2:-Unset}
 
   if [[ "$errorMessage" != Unset ]]; then
      echo -e "\\n\\nUsage Error:\\n\\n$errorMessage\\n\\n" >&2
   fi
 
   echo "USAGE for cache_purge.sh:
 
cache_purge.sh -i <SDP_INSTANCE> -f <Percentage_free>

    <Percentage_free> is an integer percentage (1-100, but realistically 1-30%)

cache_purge.sh -i 1 -f 15

   or
 
cache_purge.sh -h
"
}

SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" &> /dev/null && pwd )"

# Command Line Processing
declare SDPInstance=${SDP_INSTANCE:-Undefined}
declare -i PercentageFree=15
declare -i shiftArgs=0

set +u
while [[ $# -gt 0 ]]; do
    case $1 in
        (-h) usage -h && exit 1;;
        (-i) SDPInstance=$2; shiftArgs=1;;
        (-f) PercentageFree=$2; shiftArgs=1;;
        (-*) usage -h "Unknown command line option ($1)." && exit 1;;
    esac
 
    # Shift (modify $#) the appropriate number of times.
    shift; while [[ "$shiftArgs" -gt 0 ]]; do
        [[ $# -eq 0 ]] && usage -h "Incorrect number of arguments."
        shiftArgs=$shiftArgs-1
        shift
    done
done
set -u

if [[ $SDPInstance == Undefined ]]; then
    echo "Instance parameter not supplied."
    echo "You must supply the Perforce instance as a parameter to this script or set SDP_INSTANCE."
    exit 1
fi

. /p4/common/bin/p4_vars $SDPInstance
. /p4/common/bin/backup_functions.sh
LOGFILE=$LOGS/cache_purge.log

######### Start of Script ##########
check_vars
set_vars
rotate_log_file $LOGFILE
log "Start $P4SERVER Cache Purge"
check_uid
check_dirs
$P4CBIN/p4login

p4 cachepurge -f $PercentageFree -O -R -D //... >> $LOGFILE 2>&1

check_disk_space
remove_old_logs
log "End $P4SERVER Cache Purge"
