#!/bin/bash
#==============================================================================
# Copyright and license info is available in the LICENSE file included with
# the Server Deployment Package (SDP), and also available online:
# https://swarm.workshop.perforce.com/projects/perforce-software-sdp/view/main/LICENSE
#------------------------------------------------------------------------------
# This script automates a connectivity test for Perforce Helix Swarm triggers.
# This should be run on the Helix Core (p4d) server machine where Swarm
# triggers are installed.  It tests settings in the swarm-trigger.config
# file including the host and trigger token, as well as testing connectivity
# from the Helix Core to the Helix Swarm server. It exercises things such as
# DNS names, HTTPS certificates (if used), current availability of the Swarm
# server/service, and network firewall rules for port 80 (HTTP) or 443 (HTTPS).
#------------------------------------------------------------------------------
# The core of the test was extracted from a Perforce Support article.
# See the 'Trigger Problems' section of the Knowledge Base article
# Troubleshooting Helix Swarm: https://portal.perforce.com/s/article/3696 
#------------------------------------------------------------------------------
set -u

# Version ID Block. Relies on +k filetype modifier.
#------------------------------------------------------------------------------
# shellcheck disable=SC2016
declare VersionID='$Id: //p4-sdp/dev_c2s/Unsupported/setup/swarm_triggers_test.sh#2 $ $Change: 31472 $'
declare VersionStream=${VersionID#*//}; VersionStream=${VersionStream#*/}; VersionStream=${VersionStream%%/*};
declare VersionCL=${VersionID##*: }; VersionCL=${VersionCL%% *}
declare Version=${VersionStream}.${VersionCL}
[[ "$VersionStream" == r* ]] || Version="${Version^^}"

declare ThisScript=${0##*/}
declare ThisUser=
declare -i ErrorCount=0
declare Cmd=
declare TmpLog=
declare SwarmTriggerConf="${1:-/opt/perforce/etc/swarm-trigger.conf}"
declare SwarmTriggerScript="${2:-/opt/perforce/swarm-triggers/bin/swarm-trigger.pl}"
declare Shebang=
declare Perl=
declare WgetInsecureOption=

function msg () { echo -e "$*"; }
function errmsg () { msg "\\nError: ${1:-Unknown Error}\\n"; ErrorCount+=1; }
function bail () { errmsg "${1:-Unknown Error}"; exit "${2:-1}"; }

TmpLog=$(mktemp)
ThisUser=$(id -n -u)
msg "\\nStarting $ThisScript version $Version as $ThisUser@${HOSTNAME%%.*} at $(date)."

# shellcheck disable=SC1090
source "$SwarmTriggerConf" || bail "Could not read Swarm Trigger conf file: $SwarmTriggerConf\\nTry: $ThisScript /path/to/swarm-trigger.conf"

cd /tmp || bail "Could not do: cd /tmp"

# If the Swarm trigger conf file contains VERIFY_SSL=0, that translates into calling
# the wget script with the '--no-check-certificate' option when testing the URL. This
# is needed for self-signed certificates.
grep -q VERIFY_SSL=0 "$SwarmTriggerConf" && WgetInsecureOption="--no-check-certificate"

Cmd="wget $WgetInsecureOption --timeout 10 --post-data shelve,1234 ${SWARM_HOST}/queue/add/${SWARM_TOKEN}"

msg "Testing Swarm access via wget with:\\n\\t$Cmd"

$Cmd > "$TmpLog" 2>&1

cat "$TmpLog"

if grep -q "200 OK" "$TmpLog"; then
   msg "\\nVerified: Access with wget to Swarm was successful using settings in: $SwarmTriggerConf"
else
   errmsg "\\nDid not get expected '200 OK' output accessing Swarm using settings in: $SwarmTriggerConf"
fi

if [[ -r "$SwarmTriggerScript" ]]; then
   msg "Verified: The Swarm trigger script exists: $SwarmTriggerScript"
   if [[ -x "$SwarmTriggerScript" ]]; then
      msg "Verified: The Swarm trigger script is executable."
      Shebang=$(head -1 "$SwarmTriggerScript")
      if [[ "$Shebang" =~ ^#!* ]]; then
         Shebang="${Shebang#\#\!}"
         if [[ "$Shebang" =~ /env ]]; then
	    msg "Finding Perl based on current PATH due to 'env' in shebang directive $Shebang:"
	    Perl=$(command -v perl)
            msg "Perl detected from PATH: $Perl"
	 else
            Perl="$Shebang"
            msg "Perl detected from shebang directly is: $Perl"
	 fi
         Cmd="$Perl $SwarmTriggerScript -t ping -v 0"
         msg "Testing Swarm access via Swarm trigger script with:\\n\\t$Cmd"

         $Cmd > "$TmpLog" 2>&1
         cat "$TmpLog"

	 if grep -q "Error" "$TmpLog"; then
            errmsg "The Swarm trigger script could not connect to the Swarm server. Modifications may be needed. See the section titled 'Review cannot be updated when using triggers' in this documentation: https://www.perforce.com/manuals/v22.1/swarm/Content/Swarm/setup.validate_install.html#Review"
	 else
            msg "Verified: The Swarm trigger script connected to the Swarm server."
	 fi
      else
         errmsg "The Swarm trigger script has a malformed shebang line: $Shebang"
      fi
   else
      errmsg "The Swarm trigger script exists but is not executable: $SwarmTriggerScript"
   fi
else
   errmsg "The Swarm trigger script does not exist: $SwarmTriggerScript"
fi

if [[ "$ErrorCount" -eq 0 ]]; then
   msg "\\nThe Swarm trigger script appears to be configured correctly."
else
   errmsg "The Swarm trigger script is NOT configured correctly."
fi

exit "$ErrorCount"
