#!/bin/bash
set -u

function msg () { echo -e "$*"; }
function bail () { msg "\\nError: ${1:-Unknown Error}\\n"; exit "${2:-1}"; }

# Version ID Block. Relies on +k filetype modifier.
#------------------------------------------------------------------------------
# shellcheck disable=SC2016
declare VersionID='$Id: //p4-sdp/dev_c2s/tools/mdcu.sh#2 $ $Change: 31472 $'
declare VersionStream=${VersionID#*//}; VersionStream=${VersionStream#*/}; VersionStream=${VersionStream%%/*};
declare VersionCL=${VersionID##*: }; VersionCL=${VersionCL%% *}
declare Version=${VersionStream}.${VersionCL}
[[ "$VersionStream" == r* ]] || Version="${Version^^}"

declare Owner=perforce_software
declare Product=Unset
declare RelString=
declare RelDate=
declare IsPartial=Unset
declare VersionFile=
declare TmpDir=
declare OutFile=
declare RelString=
declare ChangeFile=
declare Changelist=
declare WorkspaceRoot=

msg "Started ${0##*/} version $Version on host ${HOSTNAME%%.*} at $(date)."

TmpDir=$(mktemp -d)
OutFile="$TmpDir/output"
ChangeFile="$TmpDir/change.p4s"

# This script assumes the workspace maps:
# //guest/perforce_software/sdp/... to the workspace root.

WorkspaceRoot="$(p4 -ztag -F %clientRoot% info)"
[[ -n "$WorkspaceRoot" ]] || bail "Could not determine workspace root dir."

Product=sdp

for branch in dev main; do
   Cmd="p4 -s opened $WorkspaceRoot/$branch/..."
   msg "Checking for any opened file in $branch branch."

   $Cmd < /dev/null > "$OutFile" 2>&1

   # Note: The status of the 'Files not opened' message changed from an error to a warning
   # somewhere along the way.
   if ! grep -qE "^(error|warning): .* - file\(s\) not opened on this client." "$OutFile"; then
      cat "$OutFile"
      bail "Files are checked out in $branch branch.  Run only when no files are checked out in main or the dev branch to promote.  Aborting."
   fi

   msg Verified: No files checked out in $branch branch.

done

BranchSpec=$Owner-$Product-dev

Cmd="p4 -s merge -n -b $BranchSpec"
msg "Doing Merge Down Preview with:\n\t$Cmd"

$Cmd < /dev/null > "$OutFile" 2>&1

# Note: The status of the 'All revisions already integrated' message changed to a warning
# somewhere along the way.
if ! grep -qE '^(error|warning): All revision\(s\) already integrated.' "$OutFile"; then
   cat "$OutFile"
   bail "Merge work required.  Aborting."
fi

msg "\\nVerified: No merge down required.  Proceeding with Copy Up."

VersionFile="$WorkspaceRoot/dev/Version"

[[ -r $VersionFile ]] || bail "Could not read version file [$VersionFile]. Aborting."

RelString=$(cat "$VersionFile")
RelString=${RelString##*MultiArch/}
RelString=${RelString%% *}
RelString=${RelString/\//\.}
RelDate=$(cat "$VersionFile")
RelDate=${RelDate##* }
RelDate=${RelDate%\.}

while [[ $IsPartial == Unset ]]; do
   echo -e -n "\\nIs this to be a partial release? (y/n): "
   read -r -e IsPartial
   IsPartial=$(echo "$IsPartial" | tr '[:lower:]' '[:upper:]')
   if [[ "$IsPartial" != Y && "$IsPartial" != N ]]; then
      msg "\\nInalid input; enter y or n only please."
      IsPartial=Unset
   fi
done

echo -e "Change: new\\n
Description:
\\tReleased SDP $RelString $RelDate." > "$ChangeFile"

if [[ "$IsPartial" == Y ]]; then
   echo -e "\\tCopy Up using 'p4 copy -r -b perforce_software-sdp-dev'," >> "$ChangeFile"
   echo -e "\\twith selective removal of work-in-progress files." >> "$ChangeFile"
else
   echo -e "\\tCopy Up using 'p4 copy -r -b perforce_software-sdp-dev'." >> "$ChangeFile"
fi

msg CF=
cat "$ChangeFile"

p4 change -i < "$ChangeFile" > "$OutFile" 2>&1

if grep -E 'Change \d+ created.' "$OutFile"; then
   # Parse the generated changelist number from the output file.
   Changelist=$(cat "$OutFile")
   Changelist=${Changelist#Change }
   Changelist=${Changelist%% *}
   [[ -n "$Changelist" ]] ||\
      bail "Could not determine pending change created."
else
   cat "$OutFile"
   bail "Failed to create pending changelist. Aborting."
fi

Cmd="p4 -s copy -c $Changelist -r -b $BranchSpec"
echo -e "Doing Copy Up with:\\n\\t$Cmd"

$Cmd < /dev/null > "$OutFile" 2>&1
cat "$OutFile"

[[ -d "$TmpDir" ]] && /bin/rm -rf "$TmpDir"

msg "To submit, do:\\n\\tp4 -s submit -c $Changelist\\n" 
